"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "default", {
    enumerable: true,
    get: function() {
        return DomainClient;
    }
});
const _errorHandler = require("./errorHandler");
let DomainClient = class DomainClient {
    /**
   * Returns domain client token
   * @returns {String} client token
   */ get token() {
        return this._token;
    }
    /**
   * Sends a MetaStats API request
   * @param {Function} getOpts function to get request options
   * @param {String} accountId account id
   * @returns {Object|String|any} request result
   */ async requestMetastats(getOpts, accountId) {
        await this._updateHost();
        await this._updateAccountHost(accountId);
        const accountCache = this._accountCache[accountId];
        try {
            const regionSettings = accountCache.regions[accountCache.regionIndex];
            const opts = getOpts(`https://metastats-api-v1.${regionSettings.region}.${this._urlCache.domain}`, regionSettings.id);
            return await this._httpClient.request(opts);
        } catch (err) {
            if (![
                "ConflictError",
                "InternalError",
                "ApiError",
                "TimeoutError"
            ].includes(err.name)) {
                throw err;
            } else {
                if (accountCache.regions.length === accountCache.regionIndex + 1) {
                    accountCache.regionIndex = 0;
                    throw err;
                } else {
                    accountCache.regionIndex++;
                    return await this.requestMetastats(getOpts, accountId);
                }
            }
        }
    }
    async _updateHost() {
        if (!this._urlCache || this._urlCache.lastUpdated < Date.now() - 1000 * 60 * 10) {
            const urlSettings = await this._httpClient.requestWithFailover({
                url: `https://mt-provisioning-api-v1.${this._domain}/users/current/servers/mt-client-api`,
                method: "GET",
                headers: {
                    "auth-token": this._token
                },
                json: true
            });
            this._urlCache = {
                domain: urlSettings.domain,
                lastUpdated: Date.now()
            };
        }
    }
    async _updateAccountHost(accountId) {
        if (!this._accountCache[accountId] || this._accountCache[accountId].lastUpdated < Date.now() - 1000 * 60 * 10) {
            const getAccount = async (id)=>{
                const accountOpts = {
                    url: `https://mt-provisioning-api-v1.${this._domain}/users/current/accounts/${id}`,
                    method: "GET",
                    headers: {
                        "auth-token": this._token
                    },
                    json: true
                };
                return await this._httpClient.requestWithFailover(accountOpts);
            };
            let accounts = [];
            let accountData = await getAccount(accountId);
            if (accountData.primaryAccountId) {
                accountData = await getAccount(accountData.primaryAccountId);
            }
            accounts = [
                {
                    _id: accountData._id,
                    region: accountData.region,
                    state: accountData.state
                }
            ].concat(accountData.accountReplicas || []);
            accounts = accounts.filter((account)=>account.state === "DEPLOYED");
            if (!accounts.length) {
                throw new _errorHandler.ValidationError("There are no replicas deployed yet. Please make sure at least " + "one of the replicas is deployed.");
            }
            let regions = accounts.map((account)=>({
                    region: account.region,
                    id: account._id
                }));
            this._accountCache[accountId] = {
                regions,
                regionIndex: 0,
                lastUpdated: Date.now()
            };
        }
    }
    /**
   * Constructs domain client instance
   * @param {HttpClient} httpClient HTTP client
   * @param {String} token authorization token
   * @param {String} domain domain to connect to, default is agiliumtrade.agiliumtrade.ai
   */ constructor(httpClient, token, domain = "agiliumtrade.agiliumtrade.ai"){
        this._httpClient = httpClient;
        this._domain = domain;
        this._token = token;
        this._urlCache = null;
        this._accountCache = {};
    }
};

//# sourceMappingURL=data:application/json;base64,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