"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    HttpClientMock: function() {
        return HttpClientMock;
    },
    default: function() {
        return HttpClient;
    }
});
const _axios = /*#__PURE__*/ _interop_require_default(require("axios"));
const _errorHandler = require("./errorHandler");
const _timeoutError = /*#__PURE__*/ _interop_require_default(require("./timeoutError"));
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let HttpClient = class HttpClient {
    /**
   * Performs a request. Response errors are returned as ApiError or subclasses.
   * @param {Object} options request options
   * @returns {Object|String|any} request result
   */ async request(options, endTime = Date.now() + this._maxRetryDelay * this._retries) {
        options.timeout = this._timeout;
        let retryAfterSeconds = 0;
        options.callback = (e, res)=>{
            if (res && res.status === 202) {
                retryAfterSeconds = res.headers["retry-after"];
            }
        };
        let body;
        try {
            const response = await this._makeRequest(options);
            options.callback(null, response);
            body = response && response.data || undefined;
        } catch (err) {
            throw this._convertError(err);
        }
        if (retryAfterSeconds) {
            await this._handleRetry(endTime, retryAfterSeconds * 1000);
            body = await this.request(options, endTime);
        }
        return body;
    }
    /**
   * Performs a request with failover. Response errors are returned as ApiError or subclasses.
   * @param {Object} options request options
   * @returns {Object|String|any} request result
   */ async requestWithFailover(options, retryCounter = 0, endTime = Date.now() + this._maxRetryDelay * this._retries) {
        options.timeout = this._timeout;
        let retryAfterSeconds = 0;
        options.callback = (e, res)=>{
            if (res && res.status === 202) {
                retryAfterSeconds = res.headers["retry-after"];
            }
        };
        let body;
        try {
            const response = await this._makeRequest(options);
            options.callback(null, response);
            body = response && response.data || undefined;
        } catch (err) {
            retryCounter = await this._handleError(err, retryCounter, endTime);
            return this.requestWithFailover(options, retryCounter, endTime);
        }
        if (retryAfterSeconds) {
            await this._handleRetry(endTime, retryAfterSeconds * 1000);
            body = await this.requestWithFailover(options, retryCounter, endTime);
        }
        return body;
    }
    _makeRequest(options) {
        return (0, _axios.default)({
            transitional: {
                clarifyTimeoutError: true
            },
            ...options
        });
    }
    async _wait(pause) {
        await new Promise((res)=>setTimeout(res, pause));
    }
    async _handleRetry(endTime, retryAfter) {
        if (endTime > Date.now() + retryAfter) {
            await this._wait(retryAfter);
        } else {
            throw new _timeoutError.default("Timed out waiting for the end of the process of calculating metrics");
        }
    }
    async _handleError(err, retryCounter, endTime) {
        const error = this._convertError(err);
        if ([
            "InternalError",
            "ApiError"
        ].includes(error.name) && retryCounter < this._retries) {
            const pause = Math.min(Math.pow(2, retryCounter) * this._minRetryDelay, this._maxRetryDelay);
            await this._wait(pause);
            return retryCounter + 1;
        } else if (error.name === "TooManyRequestsError") {
            const retryTime = Date.parse(error.metadata.recommendedRetryTime);
            if (retryTime < endTime) {
                await this._wait(retryTime - Date.now());
                return retryCounter;
            }
        }
        throw error;
    }
    // eslint-disable-next-line complexity
    _convertError(err) {
        var _err_config;
        const errorResponse = err.response || {};
        const errorData = errorResponse.data || {};
        const status = errorResponse.status || err.status;
        const url = err === null || err === void 0 ? void 0 : (_err_config = err.config) === null || _err_config === void 0 ? void 0 : _err_config.url;
        const errMsg = errorData.message || err.message;
        const errMsgDefault = errorData.message || err.code || err.message;
        switch(status){
            case 400:
                return new _errorHandler.ValidationError(errMsg, errorData.details || err.details, url);
            case 401:
                return new _errorHandler.UnauthorizedError(errMsg, url);
            case 403:
                return new _errorHandler.ForbiddenError(errMsg, url);
            case 404:
                return new _errorHandler.NotFoundError(errMsg, url);
            case 429:
                return new _errorHandler.TooManyRequestsError(errMsg, errorData.metadata || err.metadata, url);
            case 500:
                return new _errorHandler.InternalError(errMsg, url);
            default:
                return new _errorHandler.ApiError(_errorHandler.ApiError, errMsgDefault, status, url);
        }
    }
    /**
   * @typedef {Object} RetryOptions retry options
   * @property {Number} [retries] the number of attempts to retry failed request, default 5
   * @property {Number} [minDelayInSeconds] minimum delay in seconds before retrying, default 1
   * @property {Number} [maxDelayInSeconds] maximum delay in seconds before retrying, default 30
   */ /**
   * Constructs HttpClient class instance
   * @param {Number} timeout request timeout in seconds
   * @param {RetryOptions} [retryOpts] retry options
   */ constructor(timeout = 60, retryOpts = {}){
        this._timeout = timeout * 1000;
        this._retries = retryOpts.retries || 5;
        this._minRetryDelay = (retryOpts.minDelayInSeconds || 1) * 1000;
        this._maxRetryDelay = (retryOpts.maxDelayInSeconds || 30) * 1000;
    }
};
let HttpClientMock = class HttpClientMock extends HttpClient {
    _makeRequest() {
        return this._requestFn.apply(this, arguments);
    }
    /**
   * Constructs HTTP client mock
   * @param {Function(options:Object):Promise} requestFn mocked request function
   * @param {Number} timeout request timeout in seconds
   * @param {RetryOptions} retryOpts retry options
   */ constructor(requestFn, timeout, retryOpts){
        super(timeout, retryOpts);
        this._requestFn = requestFn;
    }
};

//# sourceMappingURL=data:application/json;base64,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