"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "default", {
    enumerable: true,
    get: function() {
        return MetaStatsClient;
    }
});
let MetaStatsClient = class MetaStatsClient {
    /**
   * @typedef {Object} CurrencySummaryHistoryDayMetrics profit from trading a currency pair in one trading day
   * @property {String} date date of trading day, in broker timezone, YYYY-MM-DD format
   * @property {Number} totalProfit total profit at the end of the day
   * @property {Number} [totalPips] total pips of trading day
   * @property {Number} [shortProfit] total profit of short trades per day
   * @property {Number} [longProfit] total profit of long trades per day
   * @property {Number} [shortPips] total pips of short trades per day
   * @property {Number} [longPips] total pips of long trades per day
   */ /**
   * @typedef {Object} CurrencySummaryTotalMetrics provides general data of this currency trading
   * @property {Number} profit cumulative profit of this currency trading
   * @property {Number} trades the number of all trades with this currency
   * @property {Number} [pips] cumulative pips of this currency trading
   * @property {Number} [wonTrades] the number of winning trades with this currency
   * @property {Number} [lostTrades] the number of losing trades with this currency
   * @property {Number} [wonTradesPercent] percentage of winning trades with this currency
   * @property {Number} [lostTradesPercent] percentage of losing trades with this currency
   */ /**
   * @typedef {Object} CurrencySummaryTradeMetrics provides profit and number of trades in specific trade and currency
   * @property {Number} profit cumulative profit of this currency trading
   * @property {Number} trades the number of all trades with this currency
   * @property {Number} [pips] cumulative pips of this currency trading
   */ /**
   * @typedef {Object} CurrencySummaryMetrics provides statistics on winning and losing trades indicating
   * the amount in the context of long and short positions.
   * Statistics is given for all currency pairs, for which positions were opened
   * @property {String} currency trading currency pair
   * @property {Array<CurrencySummaryHistoryDayMetrics>} history history of trading a currency pair per trading days
   * @property {CurrencySummaryTotalMetrics} total general data (such as profit, number of trades)
   * about trading a specific currency pair
   * @property {CurrencySummaryTradeMetrics} [short] profit and number of trades of short trades in a specific currency
   * @property {CurrencySummaryTradeMetrics} [long] profit and number of trades of long trades in a specific currency
   */ /**
   * @typedef {Object} PeriodMetrics provides statistics for one trade period compared to
   * the results for the previous period
   * @property {Number} [profit] cumulative profit of this period
   * @property {Number} [pips] cumulative pips of this period
   * @property {Number} [lots] cumulative lots of this period
   * @property {Number} [gain] gain of this period
   * @property {Number} [trades] the number of trades of this period
   * @property {Number} [wonTradesPercent] percentage of winning trades of this period
   * @property {Number} [profitDifference] difference in profit with the previous period
   * @property {Number} [pipsDifference] difference in pips with the previous period
   * @property {Number} [lotsDifference] difference in lots with the previous period
   * @property {Number} [gainDifference] difference in gain with the previous period
   * @property {Number} [tradesDifference] difference in the number of trades with the previous period
   * @property {Number} [wonTradesPercentDifference] difference in percentage of winning trades with the previous period
   */ /**
   * @typedef {Object} Periods provides statistics for today, this week, this month, this year
   * @property {PeriodMetrics} [today] trade information for today
   * @property {PeriodMetrics} [thisWeek] trade information for this week
   * @property {PeriodMetrics} [thisMonth] trade information for this month
   * @property {PeriodMetrics} [thisYear] trade information for this year
   */ /**
   * @typedef {Object} DailyGrowthMetrics provides each profit received from the volume of trade and changes in
   * balance, total accumulated income and existing account drawdown by day
   * @property {String} date date of trading day in broker timezone, YYYY-MM-DD format
   * @property {Number} [profit] cumulative profit per day
   * @property {Number} [pips] cumulative pips per day
   * @property {Number} [lots] cumulative lots per day
   * @property {Number} [gains] cumulative gains per day
   * @property {Number} totalProfit total profit in this day end
   * @property {Number} totalGains total gains in this day end
   * @property {Number} balance balance in this day end
   * @property {Number} [drawdownPercentage] percentage of balance drawdown in this day end
   * @property {Number} [drawdownProfit] maximum registered balance drawdown in basic currency during this day
   */ /**
   * @typedef {Object} MonthlyAnalyticCurrencyMetrics currency pair trading information for monthly analysis
   * @property {String} currency currency pair
   * @property {Number} [averageHoldingTimeLongsInMilliseconds] average holding time of long trades
   * @property {Number} [averageHoldingTimeShortsInMilliseconds] average holding time of short trades
   * @property {Number} rewardToRiskRatio the difference between reward and risk, where the lesser is always one.
   * So 0 means reward:risk=1:1, 2 means 3:1, -0.5 means 1:1.5
   * @property {Number} popularityPercent the percentage of popularity of this currency this month
   */ /**
   * @typedef {Object} MonthlyAnalyticsMetrics monthly analysis of trading on this account
   * @property {String} date date of trading month in broker timezone, YYYY-MM format
   * @property {Number} [profit] cumulative profit per month
   * @property {Number} [pips] cumulative pips per month
   * @property {Number} [lots] cumulative lots per month
   * @property {Number} [gains] cumulative gains per month
   * @property {Number} [trades] the number of trades of this month
   * @property {Array<MonthlyAnalyticCurrencyMetrics>} [currencies] list of currency pair trading
   * informations for monthly analysis
   */ /**
   * @typedef {Object} TradeByTimeMetrics opening/closing deals by days of the week or by by hours of the day
   * @property {String} date date of trading month in broker timezone, YYYY-MM format
   * @property {Number} profit the total profit of the trades at this time
   * @property {Number} [shortProfit] the total profit of short trades at this time
   * @property {Number} [longProfit] the total profit of long trades at this time
   * @property {Number} [wonProfit] the total profit of winning trades at this time
   * @property {Number} [lostProfit] the total profit of losing trades at this time
   * @property {Number} [pips] the total pips of the trades at this time
   * @property {Number} [shortPips] the total pips of short trades at this time
   * @property {Number} [longPips] the total pips of long trades at this time
   * @property {Number} [wonPips] the total pips of winning trades at this time
   * @property {Number} [lostPips] the total pips of losing trades at this time
   * @property {Number} lots cumulative lots of trades at this time
   * @property {Number} gains cumulative gains of trades at this time
   * @property {Number} [shortGains] cumulative gains of short trades at this time
   * @property {Number} [longGains] cumulative gains of long trades at this time
   * @property {Number} [wonGains] cumulative gains of winning trades at this time
   * @property {Number} [lostGains] cumulative gains of losing trades at this time
   * @property {Number} trades the number of all trades at this time
   * @property {Number} [shortTrades] the number of short trades at this time
   * @property {Number} [longTrades] the number of long trades at this time
   * @property {Number} [wonTrades] the number of winning trades at this time
   * @property {Number} [lostTrades] the number of losing trades at this time
   * @property {Number} [shortTradesPercent] percentage of short trades at this time
   * @property {Number} [longTradesPercent] percentage of long trades at this time
   * @property {Number} [wonTradesPercent] percentage of winning trades at this time
   * @property {Number} [lostTradesPercent] percentage of losing trades at this time
   * @property {Number} [hour] day hour (only for by hour case)', within 0-23
   * @property {Number} [day] weekday number (only for by day case), within 0-6
   */ /**
   * @typedef {Object} RiskOfRuinMetrics risk of ruin of balance metrics
   * @property {Number} lossSize loss size of balance
   * @property {Number} probabilityOfLoss probability of loss shows the risk of losing a particular part of the balance
   * @property {Number} consecutiveLosingTrades the number of losing trades that must be entered sequentially
   * in order for this part of the balance to be lost
   */ /**
   * @typedef {Object} OneTradeDurationMetrics metrics of one trade duration
   * @property {Array<Number>} gains list of gains for this duration
   * @property {Array<Number>} profits list of profits for this duration
   * @property {Array<Number>} lots list of lots for this duration
   * @property {Array<Number>} [pips] list of pips for this duration
   * @property {Number} durationInMinutes duration of trades in minutes
   */ /**
   * @typedef {Object} TradeDurationMetrics metrics for each duration of trades
   * @property {Array<OneTradeDurationMetrics>} [won] metrics of winning trades
   * @property {Array<OneTradeDurationMetrics>} [lost] metrics of losing trades
   */ /**
   * @typedef {Object} TradeDurationDiagramColumnCollectionMetrics collection of metrics of trades
   * in the current column for the diagram
   * @property {Array<Number>} gains list of gains
   * @property {Array<Number>} profits list of profits
   * @property {Array<Number>} lots list of lots
   * @property {Array<Number>} [pips] list of pips
   */ /**
   * @typedef {Object} TradeDurationDiagramColumnMetrics information column about
   * the duration of trades for the diagram
   * @property {Number} durations the number of durations in this column
   * @property {Number} trades the number of trades in this column
   * @property {String} name name of this column, one of 'seconds', 'minutes', 'hours', 'days', 'weeks', 'months'
   * @property {Number} minDurationInSeconds minimum trade duration in this column in seconds
   * @property {Number} [maxDurationInSeconds] maximum trade duration in this column in seconds
   * @property {TradeDurationDiagramColumnCollectionMetrics} [won] collection of metrics of winning
   * trades in this column
   * @property {TradeDurationDiagramColumnCollectionMetrics} [lost] collection of metrics of losing
   * trades in this column
   */ /**
   * @typedef {Object} Metrics trading statistics metrics
   * @property {Boolean} [inclusive] indicates whether open positions are included in the metrics,
   * "false" means that there are no open positions. Only for a request with includeOpenPositions=true
   * @property {Number} balance money on the account, not accounting for the results of currently open positions
   * @property {String} [highestBalanceDate] date of maximum balance that have ever been on the account,
   * in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {Number} [highestBalance] maximum balance that have ever been on the account
   * @property {Number} equity the result (current amount) of all positions, including opened
   * @property {Number} margin current value of margin
   * @property {Number} freeMargin current value of free margin
   * @property {Number} [marginLevel] current value of margin level 
   * @property {Number} trades total number of closed positions on the account
   * @property {Number} [withdrawals] total amount withdrawn from the deposit
   * @property {Number} [averageTradeLengthInMilliseconds] average trade length
   * (time from open to close) in milliseconds
   * @property {Number} [bestTrade] the best profit from one trade that has ever been on the account
   * @property {Number} [worstTrade] the worst profit from one trade that has ever been on the account
   * @property {Number} [bestTradePips] the best pips from one trade that has ever been on the account
   * @property {Number} [worstTradePips] the worst pips from one trade that has ever been on the account
   * @property {String} [bestTradeDate] date of the best profit from one trade that have ever been on the account,
   * in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {String} [bestTradePipsDate] date of the best pips from one trade that have ever been on the account,
   * in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {String} [worstTradeDate] date of the worst profit from one trade that have ever been on the account,
   * in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {String} [worstTradePipsDate] date of the worst pips from one trade that have ever been on the account,
   * in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {Number} [cagr] compound annual growth rate
   * @property {Number} [commissions] commissions charged by the broker for the entire period
   * @property {Number} [dailyGain] compound daily rate of return
   * @property {Number} [monthlyGain] compound monthly rate of return
   * @property {Number} [equityPercent] percentage of current equity to balance
   * @property {Number} [expectancy] the average expected profitability of one trade in basic currency
   * @property {Number} [expectancyPips] the average expected profitability of one trade in pips
   * @property {Number} [gain] time-weighted rate of return
   * @property {Number} [geometricHoldingPeriodReturn] geometric holding period return
   * @property {Number} [interest] cumulative interest and swap for the entire period
   * @property {Number} [longTrades] the number of long trades
   * @property {Number} [shortTrades] the number of short trades
   * @property {Number} [longWonTrades] the number of long winning trades
   * @property {Number} [shortWonTrades] the number of short winning trades
   * @property {Number} [longWonTradesPercent] percentage of long winning trades
   * @property {Number} [shortWonTradesPercent] percentage of short winning trades
   * @property {Number} [maxDrawdown] percentage of maximum drawdown of balance during the entire trading history
   * @property {Number} [mar] mar ratio
   * @property {Number} [lots] total volume of trades
   * @property {Number} [pips] cumulative price units
   * @property {Number} profit the total yield of closed positions for the entire period (total result)
   * @property {Number} deposits cumulative deposit for the entire period
   * @property {Number} [absoluteGain] simple deposit increase without regard to reinvestment
   * @property {Number} [profitFactor] the amount yielded by winning trades divided by the amount
   * of losses yielded by losing trades. Result in range 0 - Infinity means: `0` - only loss, `1` - profit equals to
   * loss, `Infinity` - only profit.
   * @property {Number} [sharpeRatio] average return earned in excess of the risk-free rate per unit of volatility.
   * It is calculated if there are at least 30 closed deals in the history
   * @property {Number} [sortinoRatio] differentiates harmful volatility from total overall volatility.
   * It is calculated if there are at least 30 closed deals in the history
   * @property {Number} [standardDeviationProfit] statistical measure of volatility shows how much
   * variation or dispersion. It is calculated if there are at least 30 closed deals in the history
   * @property {Number} [kurtosisProfit] a statistical measure that is used to describe profit distribution.
   * It is calculated if there are at least 30 closed deals in the history
   * @property {Number} [averageHoldingPeriodReturn] average holding period return.
   * It is calculated if there are at least 30 closed deals in the history
   * @property {Number} [averageWin] average win in basic currency
   * @property {Number} [averageWinPips] average win in pips
   * @property {Number} [averageLoss] average loss in basic currency
   * @property {Number} [averageLossPips] average loss in pips
   * @property {Number} [wonTradesPercent] percentage of winning trades
   * @property {Number} [lostTradesPercent] percentage of losing trades
   * @property {Number} [zScore] ability of a trading system to generate wins and losses in streaks.
   * It is calculated if there are at least 30 closed deals in the history
   * @property {Number} [probability] probability that a profit will be followed by a profit and a loss by a loss
   * @property {Number} [daysSinceTradingStarted] the number of days that have passed
   * since the opening of the first trade
   * @property {Array<CurrencySummaryMetrics>} [currencySummary] currency trading summary
   * @property {Array<DailyGrowthMetrics>} [dailyGrowth] daily gain shows the change
   * in account profitability on trading days
   * @property {Array<MonthlyAnalyticsMetrics>} [monthlyAnalytics] monthly analysis of trading on this account
   * @property {Array<TradeByTimeMetrics>} [closeTradesByWeekDay] closing deals by days of the week
   * @property {Array<TradeByTimeMetrics>} [openTradesByHour] opening deals by hour of the day
   * @property {Periods} [periods] trading stats for a few periods compared to the results for the previous period
   * @property {Array<RiskOfRuinMetrics>} [riskOfRuin] risk of ruin of balance
   * @property {TradeDurationMetrics} [tradeDuration] metrics for each duration of trades
   * @property {Array<TradeDurationDiagramColumnMetrics>} [tradeDurationDiagram] list of information columns about the
   * duration of trades for the diagram
   * @property {Number} [totalTradeMarketValue] total market value of all trades on the account
   */ /**
   * Returns metrics of MetaApi account. This API call is billable
   * https://metaapi.cloud/docs/metastats/restApi/api/calculateMetrics/
   * @param {String} accountId MetaApi account id
   * @param {Boolean} [includeOpenPositions] indicates whether open positions will be included
   * in the metrics, default false
   * @return {Metrics} account metrics
   */ async getMetrics(accountId, includeOpenPositions = false) {
        const getOpts = (host, id)=>({
                url: host + `/users/current/accounts/${id}/metrics`,
                method: "GET",
                headers: {
                    "auth-token": this._domainClient.token
                },
                params: {
                    includeOpenPositions
                },
                json: true
            });
        const { metrics } = await this._domainClient.requestMetastats(getOpts, accountId);
        return metrics;
    }
    /**
   * @typedef Trade historical trade
   * @property {String} _id historical trade id
   * @property {String} accountId MetaApi account id
   * @property {Number} volume trade volume
   * @property {Number} durationInMinutes trade duration in minutes
   * @property {Number} profit trade profit
   * @property {Number} gain trade gain
   * @property {String} success trade success
   * @property {String} openTime time the trade was opened at in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {String} type trade type
   * @property {String} [symbol] symbol the trade relates to
   * @property {String} [closeTime] time the trade was closed at in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {Number} [openPrice] trade opening price
   * @property {Number} [closePrice] trade closing price
   * @property {Number} [pips] the number of pips earned (positive) or lost (negative) in this trade
   * @property {Number} [riskInBalancePercent] trade risk in % of balance
   * @property {Number} [riskInPips] trade risk in pips
   * @property {String} [comment] trade comment
   */ /**
   * Returns historical trades of MetaApi account
   * https://metaapi.cloud/docs/metastats/restApi/api/getHistoricalTrades/
   * @param {String} accountId MetaApi account id
   * @param {String} startTime start of time range, inclusive
   * @param {String} endTime end of time range, exclusive
   * @param {Boolean} [updateHistory] update historical trades before returning results. 
   * If set to true, the API call will be counted towards billable MetaStats API calls. 
   * If set to false, the API call is not billable. Default is true
   * @param {Number} [limit] pagination limit
   * @param {Number} [offset] pagination offset
   * @param {Number} [marketValue ] trade market value
   * @return {Array<Trade>} account historical trades
   */ async getAccountTrades(accountId, startTime, endTime, updateHistory = true, limit = 1000, offset = 0) {
        const getOpts = (host, id)=>({
                url: host + `/users/current/accounts/${id}/historical-trades/${startTime}/${endTime}`,
                method: "GET",
                headers: {
                    "auth-token": this._domainClient.token
                },
                params: {
                    updateHistory,
                    limit,
                    offset
                },
                json: true
            });
        const { trades } = await this._domainClient.requestMetastats(getOpts, accountId);
        return trades;
    }
    /**
   * @typedef OpenTrade open trade
   * @property {String} _id historical trade id
     @property {String} accountId MetaApi account id
     @property {Number} volume trade volume
     @property {Number} durationInMinutes trade duration in minutes
     @property {Number} profit trade profit
     @property {Number} gain trade gain
     @property {String} success trade success
     @property {String} openTime time the trade was opened at in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
     @property {String} type trade type
     @property {String} symbol symbol the trade relates to
     @property {Number} openPrice trade opening price
     @property {Number} pips the number of pips earned (positive) or lost (negative) in this trade
   * @property {Number} [riskInBalancePercent] trade risk in % of balance
   * @property {Number} [riskInPips] trade risk in pips
   * @property {String} [comment] trade comment
   */ /**
   * Returns open trades of MetaApi account. This API call is not billable
   * https://metaapi.cloud/docs/metastats/restApi/api/getOpenTrades/
   * @param {String} accountId MetaApi account id
   * @param {Number} [marketValue] trade market value
   * @return {Array<OpenTrade>} account historical trades
   */ async getAccountOpenTrades(accountId) {
        const getOpts = (host, id)=>({
                url: host + `/users/current/accounts/${id}/open-trades`,
                method: "GET",
                headers: {
                    "auth-token": this._domainClient.token
                },
                json: true
            });
        const { openTrades } = await this._domainClient.requestMetastats(getOpts, accountId);
        return openTrades;
    }
    /**
   * Constructs MetaStats API client instance
   * @param {DomainClient} domainClient domain client
   */ constructor(domainClient){
        this._domainClient = domainClient;
    }
};

//# sourceMappingURL=data:application/json;base64,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