'use strict';

/**
 * Base class for API errors. Contains indication of HTTP status.
 */

Object.defineProperty(exports, "__esModule", {
  value: true
});
class ApiError extends Error {

  /**
   * ApiError constructor
   * @param {Function} clazz error name
   * @param {Object} message error message
   * @param {number} status HTTP status
   */
  constructor(clazz, message, status) {
    super(message);
    /**
     * Error name
     * @type {string}
     */
    this.name = clazz.name;
    /**
     * HTTP status code
     * @type {number}
     */
    this.status = status;
    Error.captureStackTrace(this, clazz);
  }

  /**
   * Sets error code, used for i18n
   * @param {string} code error code for i18n
   */
  set code(code) {
    this._code = code;
  }

  /**
   * Returns error code used for i18n
   * @return {string} error code
   */
  get code() {
    return this._code;
  }

  /**
   * Set message arguments for i18n
   * @param {Array<Object>} args arguments for i18n
   */
  set arguments(args) {
    this._args = args;
  }

  /**
   * Returns message arguments for i18n
   * @return {Array<Object>} message arguments for i18n
   */
  get arguments() {
    return this._args;
  }

}

exports.ApiError = ApiError; /**
                              * Throwing this error results in 404 (Not Found) HTTP response code.
                              */

class NotFoundError extends ApiError {

  /**
   * Represents NotFoundError.
   * @param {string} message error message
   */
  constructor(message) {
    super(NotFoundError, message, 404);
  }

}

exports.NotFoundError = NotFoundError; /**
                                        * Throwing this error results in 403 (Forbidden) HTTP response code.
                                        */

class ForbiddenError extends ApiError {

  /**
   * Constructs forbidden error.
   * @param {string} message error message
   */
  constructor(message) {
    super(ForbiddenError, message, 403);
  }

}

exports.ForbiddenError = ForbiddenError; /**
                                          * Throwing this error results in 401 (Unauthorized) HTTP response code.
                                          */

class UnauthorizedError extends ApiError {

  /**
   * Constructs unauthorized error.
   * @param {string} message error message
   */
  constructor(message) {
    super(UnauthorizedError, message, 401);
  }

}

exports.UnauthorizedError = UnauthorizedError; /**
                                                * Represents validation error. Throwing this error results in 400 (Bad Request) HTTP response code.
                                                */

class ValidationError extends ApiError {

  /**
   * Constructs validation error.
   * @param {string} message error message
   * @param {Object} details error data
   */
  constructor(message, details) {
    super(ValidationError, message, 400);
    /**
     * Validation error details
     * @type {Object}
     */
    this.details = details;
  }

}

exports.ValidationError = ValidationError; /**
                                            * Represents unexpected error. Throwing this error results in 500 (Internal Error) HTTP response code.
                                            */

class InternalError extends ApiError {

  /**
   * Constructs unexpected error.
   * @param {string} message error message
   */
  constructor(message) {
    super(InternalError, message, 500);
  }

}

exports.InternalError = InternalError; /**
                                        * Represents conflict error. Throwing this error results in 409 (Conflict) HTTP response code.
                                        */

class ConflictError extends ApiError {
  /**
   * Constructs conflict error.
   * @param {string} message error message
   */
  constructor(message) {
    super(ConflictError, message, 409);
  }
}

exports.ConflictError = ConflictError; /**
                                        * @typedef {Object} TooManyRequestsErrorMetadata
                                        * @property {Number} periodInMinutes throttling period in minutes
                                        * @property {Number} requestsPerPeriodAllowed available requests for periodInMinutes
                                        * @property {Date} recommendedRetryTime recommended date to retry request
                                        * @property {String} type error type
                                        */

/**
 * Represents too many requests error. Throwing this error results in 429 (Too Many Requests) HTTP response code.
 */

class TooManyRequestsError extends ApiError {
  /**
   * Constructs too many requests error.
   * @param {string} message error message
   * @param {TooManyRequestsErrorMetadata} metadata error metadata
   */
  constructor(message, metadata) {
    super(TooManyRequestsError, message, 429);
    this.metadata = metadata;
  }
}
exports.TooManyRequestsError = TooManyRequestsError;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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