'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _values = require('babel-runtime/core-js/object/values');

var _values2 = _interopRequireDefault(_values);

var _stringify = require('babel-runtime/core-js/json/stringify');

var _stringify2 = _interopRequireDefault(_stringify);

var _promise = require('babel-runtime/core-js/promise');

var _promise2 = _interopRequireDefault(_promise);

var _keys = require('babel-runtime/core-js/object/keys');

var _keys2 = _interopRequireDefault(_keys);

var _randomstring = require('randomstring');

var _randomstring2 = _interopRequireDefault(_randomstring);

var _socket = require('socket.io-client');

var _socket2 = _interopRequireDefault(_socket);

var _timeoutError = require('../timeoutError');

var _timeoutError2 = _interopRequireDefault(_timeoutError);

var _errorHandler = require('../errorHandler');

var _optionsValidator = require('../optionsValidator');

var _optionsValidator2 = _interopRequireDefault(_optionsValidator);

var _notSynchronizedError = require('./notSynchronizedError');

var _notSynchronizedError2 = _interopRequireDefault(_notSynchronizedError);

var _notConnectedError = require('./notConnectedError');

var _notConnectedError2 = _interopRequireDefault(_notConnectedError);

var _tradeError = require('./tradeError');

var _tradeError2 = _interopRequireDefault(_tradeError);

var _packetOrderer = require('./packetOrderer');

var _packetOrderer2 = _interopRequireDefault(_packetOrderer);

var _synchronizationThrottler = require('./synchronizationThrottler');

var _synchronizationThrottler2 = _interopRequireDefault(_synchronizationThrottler);

var _subscriptionManager = require('./subscriptionManager');

var _subscriptionManager2 = _interopRequireDefault(_subscriptionManager);

var _logger = require('../../logger');

var _logger2 = _interopRequireDefault(_logger);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

let PacketLogger;
if (typeof window === 'undefined') {
  // don't import PacketLogger for browser version
  PacketLogger = require('./packetLogger').default;
}

/**
 * MetaApi websocket API client (see https://metaapi.cloud/docs/client/websocket/overview/)
 */
class MetaApiWebsocketClient {

  /**
   * Constructs MetaApi websocket API client instance
   * @param {HttpClient} httpClient HTTP client
   * @param {String} token authorization token
   * @param {Object} opts websocket client options
   */
  // eslint-disable-next-line complexity,max-statements
  constructor(httpClient, token, opts) {
    const validator = new _optionsValidator2.default();
    opts = opts || {};
    opts.packetOrderingTimeout = validator.validateNonZero(opts.packetOrderingTimeout, 60, 'packetOrderingTimeout');
    opts.synchronizationThrottler = opts.synchronizationThrottler || {};
    this._httpClient = httpClient;
    this._application = opts.application || 'MetaApi';
    this._domain = opts.domain || 'agiliumtrade.agiliumtrade.ai';
    this._region = opts.region;
    this._hostname = 'mt-client-api-v1';
    this._url = `https://${this._hostname}.${this._domain}`;
    this._requestTimeout = validator.validateNonZero(opts.requestTimeout, 60, 'requestTimeout') * 1000;
    this._connectTimeout = validator.validateNonZero(opts.connectTimeout, 60, 'connectTimeout') * 1000;
    const retryOpts = opts.retryOpts || {};
    this._retries = validator.validateNumber(retryOpts.retries, 5, 'retryOpts.retries');
    this._minRetryDelayInSeconds = validator.validateNonZero(retryOpts.minDelayInSeconds, 1, 'retryOpts.minDelayInSeconds');
    this._maxRetryDelayInSeconds = validator.validateNonZero(retryOpts.maxDelayInSeconds, 30, 'retryOpts.maxDelayInSeconds');
    this._maxAccountsPerInstance = 100;
    this._subscribeCooldownInSeconds = validator.validateNonZero(retryOpts.subscribeCooldownInSeconds, 600, 'retryOpts.subscribeCooldownInSeconds');
    this._sequentialEventProcessing = true;
    this._useSharedClientApi = validator.validateBoolean(opts.useSharedClientApi, false, 'useSharedClientApi');
    this._unsubscribeThrottlingInterval = validator.validateNonZero(opts.unsubscribeThrottlingIntervalInSeconds, 10, 'unsubscribeThrottlingIntervalInSeconds') * 1000;
    this._token = token;
    this._synchronizationListeners = {};
    this._latencyListeners = [];
    this._reconnectListeners = [];
    this._connectedHosts = {};
    this._socketInstances = [];
    this._socketInstancesByAccounts = {};
    this._synchronizationThrottlerOpts = opts.synchronizationThrottler;
    this._subscriptionManager = new _subscriptionManager2.default(this);
    this._statusTimers = {};
    this._eventQueues = {};
    this._synchronizationFlags = {};
    this._subscribeLock = null;
    this._firstConnect = true;
    this._lastRequestsTime = {};
    this._packetOrderer = new _packetOrderer2.default(this, opts.packetOrderingTimeout);
    if (opts.packetLogger && opts.packetLogger.enabled) {
      this._packetLogger = new PacketLogger(opts.packetLogger);
      this._packetLogger.start();
    }
    this._logger = _logger2.default.getLogger('MetaApiWebsocketClient');
  }

  /**
   * Restarts the account synchronization process on an out of order packet
   * @param {String} accountId account id
   * @param {Number} instanceIndex instance index
   * @param {Number} expectedSequenceNumber expected s/n
   * @param {Number} actualSequenceNumber actual s/n
   * @param {Object} packet packet data
   * @param {Date} receivedAt time the packet was received at
   */
  onOutOfOrderPacket(accountId, instanceIndex, expectedSequenceNumber, actualSequenceNumber, packet, receivedAt) {
    if (this._subscriptionManager.isSubscriptionActive(accountId)) {
      this._logger.error('MetaApi websocket client received an out of order ' + `packet type ${packet.type} for account id ${accountId}:${instanceIndex}. Expected s/n ` + `${expectedSequenceNumber} does not match the actual of ${actualSequenceNumber}`);
      this.ensureSubscribe(accountId, instanceIndex);
    }
  }

  /**
   * Patch server URL for use in unit tests
   * @param {String} url patched server URL
   */
  set url(url) {
    this._url = url;
  }

  /**
   * Returns the list of socket instance dictionaries
   * @return {Object[]} list of socket instance dictionaries
   */
  get socketInstances() {
    return this._socketInstances;
  }

  /**
   * Returns the dictionary of socket instances by account ids
   * @return {Object} dictionary of socket instances by account ids
   */
  get socketInstancesByAccounts() {
    return this._socketInstancesByAccounts;
  }

  /**
   * Returns the list of subscribed account ids
   * @param {String} socketInstanceIndex socket instance index
   * @return {string[]} list of subscribed account ids
   */
  subscribedAccountIds(socketInstanceIndex) {
    const connectedIds = [];
    (0, _keys2.default)(this._connectedHosts).forEach(instanceId => {
      const accountId = instanceId.split(':')[0];
      if (!connectedIds.includes(accountId) && this._socketInstancesByAccounts[accountId] !== undefined && (this._socketInstancesByAccounts[accountId] === socketInstanceIndex || socketInstanceIndex === undefined)) {
        connectedIds.push(accountId);
      }
    });
    return connectedIds;
  }

  /**
   * Returns websocket client connection status
   * @param socketInstanceIndex socket instance index
   * @returns {Boolean} websocket client connection status
   */
  connected(socketInstanceIndex) {
    const instance = this._socketInstances.length > socketInstanceIndex ? this._socketInstances[socketInstanceIndex] : null;
    return instance && instance.socket && instance.socket.connected || false;
  }

  /**
   * Returns list of accounts assigned to instance
   * @param socketInstanceIndex socket instance index
   * @returns 
   */
  getAssignedAccounts(socketInstanceIndex) {
    const accountIds = [];
    (0, _keys2.default)(this._socketInstancesByAccounts).forEach(key => {
      if (this._socketInstancesByAccounts[key] === socketInstanceIndex) {
        accountIds.push(key);
      }
    });
    return accountIds;
  }

  /**
   * Locks subscription for a socket instance based on TooManyRequestsError metadata
   * @param socketInstanceIndex socket instance index
   * @param metadata TooManyRequestsError metadata
   */
  async lockSocketInstance(socketInstanceIndex, metadata) {
    if (metadata.type === 'LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_USER') {
      this._subscribeLock = {
        recommendedRetryTime: metadata.recommendedRetryTime,
        lockedAtAccounts: this.subscribedAccountIds().length,
        lockedAtTime: Date.now()
      };
    } else {
      const subscribedAccounts = this.subscribedAccountIds(socketInstanceIndex);
      if (subscribedAccounts.length === 0) {
        const socketInstance = this.socketInstances[socketInstanceIndex];
        socketInstance.socket.close();
        await this._reconnect(socketInstanceIndex);
      } else {
        const instance = this._socketInstances[socketInstanceIndex];
        instance.subscribeLock = {
          recommendedRetryTime: metadata.recommendedRetryTime,
          type: metadata.type,
          lockedAtAccounts: subscribedAccounts.length
        };
      }
    }
  }

  /**
   * Connects to MetaApi server via socket.io protocol
   * @returns {Promise} promise which resolves when connection is established
   */
  async connect() {
    let clientId = Math.random();
    let resolve;
    let result = new _promise2.default((res, rej) => {
      resolve = res;
    });
    const socketInstanceIndex = this._socketInstances.length;
    const instance = {
      id: socketInstanceIndex,
      connected: false,
      requestResolves: {},
      resolved: false,
      connectResult: result,
      sessionId: _randomstring2.default.generate(32),
      isReconnecting: false,
      socket: null,
      synchronizationThrottler: new _synchronizationThrottler2.default(this, socketInstanceIndex, this._synchronizationThrottlerOpts),
      subscribeLock: null
    };
    instance.connected = true;
    this._socketInstances.push(instance);
    instance.synchronizationThrottler.start();
    const serverUrl = await this._getServerUrl(socketInstanceIndex);
    const socketInstance = (0, _socket2.default)(serverUrl, {
      path: '/ws',
      reconnection: true,
      reconnectionDelay: 1000,
      reconnectionDelayMax: 5000,
      reconnectionAttempts: Infinity,
      timeout: this._connectTimeout,
      extraHeaders: {
        'Client-Id': clientId
      },
      query: {
        'auth-token': this._token,
        clientId: clientId,
        protocol: 2
      }
    });
    instance.socket = socketInstance;
    if (this._socketInstances.length === 1) {
      this._packetOrderer.start();
    }
    socketInstance.on('connect', async () => {
      // eslint-disable-next-line no-console
      this._logger.info('MetaApi websocket client connected to the MetaApi server');
      instance.isReconnecting = false;
      if (!instance.resolved) {
        instance.resolved = true;
        resolve();
      } else {
        await this._fireReconnected(instance.id);
      }
      if (!instance.connected) {
        instance.socket.close();
      }
    });
    socketInstance.on('reconnect', async () => {
      instance.isReconnecting = false;
      await this._fireReconnected(instance.id);
    });
    socketInstance.on('connect_error', async err => {
      // eslint-disable-next-line no-console
      this._logger.error('MetaApi websocket client connection error', err);
      instance.isReconnecting = false;
      if (!instance.resolved) {
        await this._reconnect(instance.id);
      }
    });
    socketInstance.on('connect_timeout', async timeout => {
      // eslint-disable-next-line no-console
      this._logger.error('MetaApi websocket client connection timeout');
      instance.isReconnecting = false;
      if (!instance.resolved) {
        await this._reconnect(instance.id);
      }
    });
    socketInstance.on('disconnect', async reason => {
      instance.synchronizationThrottler.onDisconnect();
      // eslint-disable-next-line no-console
      this._logger.info('MetaApi websocket client disconnected from the MetaApi ' + 'server because of ' + reason);
      instance.isReconnecting = false;
      await this._reconnect(instance.id);
    });
    socketInstance.on('error', async error => {
      // eslint-disable-next-line no-console
      this._logger.error('MetaApi websocket client error', error);
      instance.isReconnecting = false;
      await this._reconnect(instance.id);
    });
    socketInstance.on('response', data => {
      if (typeof data === 'string') {
        data = JSON.parse(data);
      }
      this._logger.debug(() => `${data.accountId}: Response received: ${(0, _stringify2.default)({
        requestId: data.requestId, timestamps: data.timestamps })}`);
      let requestResolve = instance.requestResolves[data.requestId] || { resolve: () => {}, reject: () => {} };
      delete instance.requestResolves[data.requestId];
      this._convertIsoTimeToDate(data);
      requestResolve.resolve(data);
      if (data.timestamps && requestResolve.type) {
        data.timestamps.clientProcessingFinished = new Date();
        for (let listener of this._latencyListeners) {
          _promise2.default.resolve().then(() => requestResolve.type === 'trade' ? listener.onTrade(data.accountId, data.timestamps) : listener.onResponse(data.accountId, requestResolve.type, data.timestamps)).catch(error => this._logger.error('Failed to process onResponse event for account ' + data.accountId + ', request type ' + requestResolve.type, error));
        }
      }
    });
    socketInstance.on('processingError', data => {
      let requestResolve = instance.requestResolves[data.requestId] || { resolve: () => {}, reject: () => {} };
      delete instance.requestResolves[data.requestId];
      requestResolve.reject(this._convertError(data));
    });
    socketInstance.on('synchronization', async data => {
      if (typeof data === 'string') {
        data = JSON.parse(data);
      }
      this._logger.trace(() => `${data.accountId}:${data.instanceIndex}: Sync packet received: ${(0, _stringify2.default)({
        type: data.type, sequenceNumber: data.sequenceNumber, sequenceTimestamp: data.sequenceTimestamp,
        synchronizationId: data.synchronizationId, application: data.application, host: data.host })}`);
      let activeSynchronizationIds = instance.synchronizationThrottler.activeSynchronizationIds;
      if (!data.synchronizationId || activeSynchronizationIds.includes(data.synchronizationId)) {
        if (this._packetLogger) {
          await this._packetLogger.logPacket(data);
        }
        if (!this._subscriptionManager.isSubscriptionActive(data.accountId) && data.type !== 'disconnected') {
          if (this._throttleRequest('unsubscribe', data.accountId, this._unsubscribeThrottlingInterval)) {
            this.unsubscribe(data.accountId).catch(err => {
              this._logger.warn(`${data.accountId}:${data.instanceIndex || 0}: failed to unsubscribe`, err);
            });
          }
          return;
        }
        this._convertIsoTimeToDate(data);
      } else {
        data.type = 'noop';
      }
      this.queuePacket(data);
    });
    return result;
  }

  /**
   * Closes connection to MetaApi server
   */
  close() {
    this._socketInstances.forEach(async instance => {
      if (instance.connected) {
        instance.connected = false;
        await instance.socket.close();
        for (let requestResolve of (0, _values2.default)(instance.requestResolves)) {
          requestResolve.reject(new Error('MetaApi connection closed'));
        }
        instance.requestResolves = {};
      }
    });
    this._synchronizationListeners = {};
    this._latencyListeners = [];
    this._socketInstancesByAccounts = {};
    this._socketInstances = [];
    this._packetOrderer.stop();
  }

  /**
   * MetaTrader account information (see https://metaapi.cloud/docs/client/models/metatraderAccountInformation/)
   * @typedef {Object} MetatraderAccountInformation
   * @property {String} platform platform id (mt4 or mt5)
   * @property {String} broker broker name
   * @property {String} currency account base currency ISO code
   * @property {String} server broker server name
   * @property {Number} balance account balance
   * @property {Number} equity account liquidation value
   * @property {Number} margin used margin
   * @property {Number} freeMargin free margin
   * @property {Number} leverage account leverage coefficient
   * @property {Number} marginLevel margin level calculated as % of equity/margin
   * @property {Boolean} tradeAllowed flag indicating that trading is allowed
   * @property {Boolean} [investorMode] flag indicating that investor password was used (supported for g2 only)
   * @property {String} marginMode margin calculation mode, one of ACCOUNT_MARGIN_MODE_EXCHANGE,
   * ACCOUNT_MARGIN_MODE_RETAIL_NETTING, ACCOUNT_MARGIN_MODE_RETAIL_HEDGING
   * @property {String} name Account owner name
   * @property {Number} login Account login
   * @property {Number} credit Account credit in the deposit currency
   */

  /**
   * Returns account information for a specified MetaTrader account (see
   * https://metaapi.cloud/docs/client/websocket/api/readTradingTerminalState/readAccountInformation/).
   * @param {String} accountId id of the MetaTrader account to return information for
   * @returns {Promise<MetatraderAccountInformation>} promise resolving with account information
   */
  async getAccountInformation(accountId) {
    let response = await this.rpcRequest(accountId, { application: 'RPC', type: 'getAccountInformation' });
    return response.accountInformation;
  }

  /**
   * Stop loss threshold
   * @typedef {Object} StopLossThreshold
   * @property {Number} threshold price threshold relative to position open price, interpreted according to units
   * field value
   * @property {Number} stopLoss stop loss value, interpreted according to units and basePrice field values
   */

  /**
   * Threshold trailing stop loss configuration
   * @typedef {Object} ThresholdTrailingStopLoss
   * @property {StopLossThreshold[]} thresholds stop loss thresholds
   * @property {String} [units] threshold stop loss units. ABSOLUTE_PRICE means the that the value of stop loss
   * threshold fields contain a final threshold & stop loss value. RELATIVE* means that the threshold fields value
   * contains relative threshold & stop loss values, expressed either in price, points, pips, account currency or
   * balance percentage. Default is ABSOLUTE_PRICE. One of ABSOLUTE_PRICE, RELATIVE_PRICE, RELATIVE_POINTS,
   * RELATIVE_PIPS, RELATIVE_CURRENCY, RELATIVE_BALANCE_PERCENTAGE
   * @property {String} [stopPriceBase] defined the base price to calculate SL relative to for POSITION_MODIFY and
   * pending order requests. Default is OPEN_PRICE. One of CURRENT_PRICE, OPEN_PRICE
   */

  /**
   * Distance trailing stop loss configuration
   * @typedef {Object} DistanceTrailingStopLoss
   * @property {Number} [distance] SL distance relative to current price, interpreted according to units field value
   * @property {String} [units] distance trailing stop loss units. RELATIVE_* means that the distance field value 
   * contains relative stop loss expressed either in price, points, pips, account currency or balance percentage. 
   * Default is RELATIVE_PRICE. One of RELATIVE_PRICE, RELATIVE_POINTS, RELATIVE_PIPS, RELATIVE_CURRENCY,
   * RELATIVE_BALANCE_PERCENTAGE
   */

  /**
   * Distance trailing stop loss configuration
   * @typedef {Object} TrailingStopLoss
   * @property {DistanceTrailingStopLoss} [distance] distance trailing stop loss configuration. If both distance and
   * threshold TSL are set, then the resulting SL will be the one which is closest to the current price
   * @property {ThresholdTrailingStopLoss} [threshold] distance trailing stop loss configuration. If both distance and
   * threshold TSL are set, then the resulting SL will be the one which is closest to the current price
   */

  /**
   * MetaTrader position
   * @typedef {Object} MetatraderPosition
   * @property {Number} id position id (ticket number)
   * @property {String} type position type (one of POSITION_TYPE_BUY, POSITION_TYPE_SELL)
   * @property {String} symbol position symbol
   * @property {Number} magic position magic number, identifies the EA which opened the position
   * @property {Date} time time position was opened at
   * @property {String} brokerTime time position was opened at, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {Date} updateTime last position modification time
   * @property {Number} openPrice position open price
   * @property {Number} currentPrice current price
   * @property {Number} currentTickValue current tick value
   * @property {Number} [stopLoss] optional position stop loss price
   * @property {Number} [takeProfit] optional position take profit price
   * @property {TrailingStopLoss} [trailingStopLoss] distance trailing stop loss configuration
   * @property {Number} volume position volume
   * @property {Number} swap position cumulative swap
   * @property {Number} profit position cumulative profit
   * @property {String} [comment] optional position comment. The sum of the line lengths of the comment and the clientId
   * must be less than or equal to 26. For more information see https://metaapi.cloud/docs/client/clientIdUsage/
   * @property {String} [clientId] optional client-assigned id. The id value can be assigned when submitting a trade and
   * will be present on position, history orders and history deals related to the trade. You can use this field to bind
   * your trades to objects in your application and then track trade progress. The sum of the line lengths of the
   * comment and the clientId must be less than or equal to 26. For more information see
   * https://metaapi.cloud/docs/client/clientIdUsage/
   * @property {Number} unrealizedProfit profit of the part of the position which is not yet closed, including swap
   * @property {Number} realizedProfit profit of the already closed part, including commissions and swap
   * @property {Number} commission position commission
   * @property {String} reason position opening reason. One of POSITION_REASON_CLIENT, POSITION_REASON_EXPERT,
   * POSITION_REASON_MOBILE, POSITION_REASON_WEB, POSITION_REASON_UNKNOWN. See
   * https://www.mql5.com/en/docs/constants/tradingconstants/positionproperties#enum_position_reason',
   * @property {Number} [accountCurrencyExchangeRate] current exchange rate of account currency into account base
   * currency (USD if you did not override it)
   * @property {String} [brokerComment] current comment value on broker side (possibly overriden by the broker)
   */

  /**
   * Returns positions for a specified MetaTrader account (see
   * https://metaapi.cloud/docs/client/websocket/api/readTradingTerminalState/readPositions/).
   * @param {String} accountId id of the MetaTrader account to return information for
   * @returns {Promise<Array<MetatraderPosition>} promise resolving with array of open positions
   */
  async getPositions(accountId) {
    let response = await this.rpcRequest(accountId, { application: 'RPC', type: 'getPositions' });
    return response.positions;
  }

  /**
   * Returns specific position for a MetaTrader account (see
   * https://metaapi.cloud/docs/client/websocket/api/readTradingTerminalState/readPosition/).
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {String} positionId position id
   * @return {Promise<MetatraderPosition>} promise resolving with MetaTrader position found
   */
  async getPosition(accountId, positionId) {
    let response = await this.rpcRequest(accountId, { application: 'RPC', type: 'getPosition', positionId });
    return response.position;
  }

  /**
   * MetaTrader order
   * @typedef {Object} MetatraderOrder
   * @property {Number} id order id (ticket number)
   * @property {String} type order type (one of ORDER_TYPE_SELL, ORDER_TYPE_BUY, ORDER_TYPE_BUY_LIMIT,
   * ORDER_TYPE_SELL_LIMIT, ORDER_TYPE_BUY_STOP, ORDER_TYPE_SELL_STOP). See
   * https://www.mql5.com/en/docs/constants/tradingconstants/orderproperties#enum_order_type
   * @property {String} state order state one of (ORDER_STATE_STARTED, ORDER_STATE_PLACED, ORDER_STATE_CANCELED,
   * ORDER_STATE_PARTIAL, ORDER_STATE_FILLED, ORDER_STATE_REJECTED, ORDER_STATE_EXPIRED, ORDER_STATE_REQUEST_ADD,
   * ORDER_STATE_REQUEST_MODIFY, ORDER_STATE_REQUEST_CANCEL). See
   * https://www.mql5.com/en/docs/constants/tradingconstants/orderproperties#enum_order_state
   * @property {Number} magic order magic number, identifies the EA which created the order
   * @property {Date} time time order was created at
   * @property {String} brokerTime time time order was created at, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {Date} [doneTime] time order was executed or canceled at. Will be specified for
   * completed orders only
   * @property {String} [doneBrokerTime] time order was executed or canceled at, in broker timezone,
   * YYYY-MM-DD HH:mm:ss.SSS format. Will be specified for completed orders only
   * @property {String} symbol order symbol
   * @property {Number} openPrice order open price (market price for market orders, limit price for limit orders or stop
   * price for stop orders)
   * @property {Number} [currentPrice] current price, filled for pending orders only. Not filled for history orders.
   * @property {Number} [stopLoss] order stop loss price
   * @property {Number} [takeProfit] order take profit price
   * @property {TrailingStopLoss} [trailingStopLoss] distance trailing stop loss configuration
   * @property {Number} volume order requested quantity
   * @property {Number} currentVolume order remaining quantity, i.e. requested quantity - filled quantity
   * @property {String} positionId order position id. Present only if the order has a position attached to it
   * @property {String} [comment] order comment. The sum of the line lengths of the comment and the clientId
   * must be less than or equal to 26. For more information see https://metaapi.cloud/docs/client/clientIdUsage/
   * @property {String} [brokerComment] current comment value on broker side (possibly overriden by the broker)
   * @property {String} [clientId] client-assigned id. The id value can be assigned when submitting a trade and
   * will be present on position, history orders and history deals related to the trade. You can use this field to bind
   * your trades to objects in your application and then track trade progress. The sum of the line lengths of the
   * comment and the clientId must be less than or equal to 26. For more information see
   * https://metaapi.cloud/docs/client/clientIdUsage/
   * @property {String} platform platform id (mt4 or mt5)
   * @property {String} reason order opening reason. One of ORDER_REASON_CLIENT, ORDER_REASON_MOBILE, ORDER_REASON_WEB,
   * ORDER_REASON_EXPERT, ORDER_REASON_SL, ORDER_REASON_TP, ORDER_REASON_SO, ORDER_REASON_UNKNOWN. See
   * https://www.mql5.com/en/docs/constants/tradingconstants/orderproperties#enum_order_reason.
   * @property {String} fillingMode order filling mode. One of ORDER_FILLING_FOK, ORDER_FILLING_IOC,
   * ORDER_FILLING_RETURN. See
   * https://www.mql5.com/en/docs/constants/tradingconstants/orderproperties#enum_order_type_filling.
   * @property {String} expirationType order expiration type. One of ORDER_TIME_GTC, ORDER_TIME_DAY,
   * ORDER_TIME_SPECIFIED, ORDER_TIME_SPECIFIED_DAY. See
   * https://www.mql5.com/en/docs/constants/tradingconstants/orderproperties#enum_order_type_time
   * @property {Date} expirationTime optional order expiration time
   * @property {Number} [accountCurrencyExchangeRate] current exchange rate of account currency into account base
   * currency (USD if you did not override it)
   * @property {String} [closeByPositionId] identifier of an opposite position used for closing by order
   * ORDER_TYPE_CLOSE_BY
   * @property {Number} [stopLimitPrice] the Limit order price for the StopLimit order
   */

  /**
   * Returns open orders for a specified MetaTrader account (see
   * https://metaapi.cloud/docs/client/websocket/api/readTradingTerminalState/readOrders/).
   * @param {String} accountId id of the MetaTrader account to return information for
   * @return {Promise<Array<MetatraderOrder>>} promise resolving with open MetaTrader orders
   */
  async getOrders(accountId) {
    let response = await this.rpcRequest(accountId, { application: 'RPC', type: 'getOrders' });
    return response.orders;
  }

  /**
   * Returns specific open order for a MetaTrader account (see
   * https://metaapi.cloud/docs/client/websocket/api/readTradingTerminalState/readOrder/).
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {String} orderId order id (ticket number)
   * @return {Promise<MetatraderOrder>} promise resolving with metatrader order found
   */
  async getOrder(accountId, orderId) {
    let response = await this.rpcRequest(accountId, { application: 'RPC', type: 'getOrder', orderId });
    return response.order;
  }

  /**
   * MetaTrader history orders search query response
   * @typedef {Object} MetatraderHistoryOrders
   * @property {Array<MetatraderOrder>} historyOrders array of history orders returned
   * @property {Boolean} synchronizing flag indicating that history order initial synchronization is still in progress
   * and thus search results may be incomplete
   */

  /**
   * Returns the history of completed orders for a specific ticket number (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveHistoricalData/readHistoryOrdersByTicket/).
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {String} ticket ticket number (order id)
   * @returns {Promise<MetatraderHistoryOrders>} promise resolving with request results containing history orders found
   */
  async getHistoryOrdersByTicket(accountId, ticket) {
    let response = await this.rpcRequest(accountId, { application: 'RPC', type: 'getHistoryOrdersByTicket', ticket });
    return {
      historyOrders: response.historyOrders,
      synchronizing: response.synchronizing
    };
  }

  /**
   * Returns the history of completed orders for a specific position id (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveHistoricalData/readHistoryOrdersByPosition/)
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {String} positionId position id
   * @returns {Promise<MetatraderHistoryOrders>} promise resolving with request results containing history orders found
   */
  async getHistoryOrdersByPosition(accountId, positionId) {
    let response = await this.rpcRequest(accountId, { application: 'RPC', type: 'getHistoryOrdersByPosition',
      positionId });
    return {
      historyOrders: response.historyOrders,
      synchronizing: response.synchronizing
    };
  }

  /**
   * Returns the history of completed orders for a specific time range (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveHistoricalData/readHistoryOrdersByTimeRange/)
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {Date} startTime start of time range, inclusive
   * @param {Date} endTime end of time range, exclusive
   * @param {Number} offset pagination offset, default is 0
   * @param {Number} limit pagination limit, default is 1000
   * @returns {Promise<MetatraderHistoryOrders>} promise resolving with request results containing history orders found
   */
  async getHistoryOrdersByTimeRange(accountId, startTime, endTime, offset = 0, limit = 1000) {
    let response = await this.rpcRequest(accountId, { application: 'RPC', type: 'getHistoryOrdersByTimeRange',
      startTime, endTime, offset, limit });
    return {
      historyOrders: response.historyOrders,
      synchronizing: response.synchronizing
    };
  }

  /**
   * MetaTrader history deals search query response
   * @typedef {Object} MetatraderDeals
   * @property {Array<MetatraderDeal>} deals array of history deals returned
   * @property {Boolean} synchronizing flag indicating that deal initial synchronization is still in progress
   * and thus search results may be incomplete
   */

  /**
   * MetaTrader deal
   * @typedef {Object} MetatraderDeal
   * @property {String} id deal id (ticket number)
   * @property {String} type deal type (one of DEAL_TYPE_BUY, DEAL_TYPE_SELL, DEAL_TYPE_BALANCE, DEAL_TYPE_CREDIT,
   * DEAL_TYPE_CHARGE, DEAL_TYPE_CORRECTION, DEAL_TYPE_BONUS, DEAL_TYPE_COMMISSION, DEAL_TYPE_COMMISSION_DAILY,
   * DEAL_TYPE_COMMISSION_MONTHLY, DEAL_TYPE_COMMISSION_AGENT_DAILY, DEAL_TYPE_COMMISSION_AGENT_MONTHLY,
   * DEAL_TYPE_INTEREST, DEAL_TYPE_BUY_CANCELED, DEAL_TYPE_SELL_CANCELED, DEAL_DIVIDEND, DEAL_DIVIDEND_FRANKED,
   * DEAL_TAX). See https://www.mql5.com/en/docs/constants/tradingconstants/dealproperties#enum_deal_type
   * @property {String} entryType deal entry type (one of DEAL_ENTRY_IN, DEAL_ENTRY_OUT, DEAL_ENTRY_INOUT,
   * DEAL_ENTRY_OUT_BY). See https://www.mql5.com/en/docs/constants/tradingconstants/dealproperties#enum_deal_entry
   * @property {String} [symbol] symbol deal relates to
   * @property {Number} [magic] deal magic number, identifies the EA which initiated the deal
   * @property {Date} time time the deal was conducted at
   * @property {String} brokerTime time time the deal was conducted at, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {Number} [volume] deal volume
   * @property {Number} [price] the price the deal was conducted at
   * @property {Number} [commission] deal commission
   * @property {Number} [swap] deal swap
   * @property {Number} profit deal profit
   * @property {String} [positionId] id of position the deal relates to
   * @property {String} [orderId] id of order the deal relates to
   * @property {String} [comment] deal comment. The sum of the line lengths of the comment and the clientId
   * must be less than or equal to 26. For more information see https://metaapi.cloud/docs/client/clientIdUsage/
   * @property {String} [brokerComment] current comment value on broker side (possibly overriden by the broker)
   * @property {String} [clientId] client-assigned id. The id value can be assigned when submitting a trade and
   * will be present on position, history orders and history deals related to the trade. You can use this field to bind
   * your trades to objects in your application and then track trade progress. The sum of the line lengths of the
   * comment and the clientId must be less than or equal to 26. For more information see
   * https://metaapi.cloud/docs/client/clientIdUsage/
   * @property {String} platform platform id (mt4 or mt5)
   * @property {String} [reason] optional deal execution reason. One of DEAL_REASON_CLIENT, DEAL_REASON_MOBILE,
   * DEAL_REASON_WEB, DEAL_REASON_EXPERT, DEAL_REASON_SL, DEAL_REASON_TP, DEAL_REASON_SO, DEAL_REASON_ROLLOVER,
   * DEAL_REASON_VMARGIN, DEAL_REASON_SPLIT, DEAL_REASON_UNKNOWN. See
   * https://www.mql5.com/en/docs/constants/tradingconstants/dealproperties#enum_deal_reason.
   * @property {Number} [accountCurrencyExchangeRate] current exchange rate of account currency into account base
   * currency (USD if you did not override it)
   */

  /**
   * Returns history deals with a specific ticket number (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveHistoricalData/readDealsByTicket/).
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {String} ticket ticket number (deal id for MT5 or order id for MT4)
   * @returns {Promise<MetatraderDeals>} promise resolving with request results containing deals found
   */
  async getDealsByTicket(accountId, ticket) {
    let response = await this.rpcRequest(accountId, { application: 'RPC', type: 'getDealsByTicket', ticket });
    return {
      deals: response.deals,
      synchronizing: response.synchronizing
    };
  }

  /**
   * Returns history deals for a specific position id (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveHistoricalData/readDealsByPosition/).
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {String} positionId position id
   * @returns {Promise<MetatraderDeals>} promise resolving with request results containing deals found
   */
  async getDealsByPosition(accountId, positionId) {
    let response = await this.rpcRequest(accountId, { application: 'RPC', type: 'getDealsByPosition', positionId });
    return {
      deals: response.deals,
      synchronizing: response.synchronizing
    };
  }

  /**
   * Returns history deals with for a specific time range (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveHistoricalData/readDealsByTimeRange/).
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {Date} startTime start of time range, inclusive
   * @param {Date} endTime end of time range, exclusive
   * @param {Number} offset pagination offset, default is 0
   * @param {Number} limit pagination limit, default is 1000
   * @returns {Promise<MetatraderDeals>} promise resolving with request results containing deals found
   */
  async getDealsByTimeRange(accountId, startTime, endTime, offset = 0, limit = 1000) {
    let response = await this.rpcRequest(accountId, { application: 'RPC', type: 'getDealsByTimeRange', startTime,
      endTime, offset, limit });
    return {
      deals: response.deals,
      synchronizing: response.synchronizing
    };
  }

  /**
   * Clears the order and transaction history of a specified application so that it can be synchronized from scratch
   * (see https://metaapi.cloud/docs/client/websocket/api/removeHistory/).
   * @param {String} accountId id of the MetaTrader account to remove history for
   * @param {String} [application] application to remove history for
   * @return {Promise} promise resolving when the history is cleared
   */
  removeHistory(accountId, application) {
    return this.rpcRequest(accountId, { application, type: 'removeHistory' });
  }

  /**
   * Clears the order and transaction history of a specified application and removes the application (see
   * https://metaapi.cloud/docs/client/websocket/api/removeApplication/).
   * @param {String} accountId id of the MetaTrader account to remove history and application for
   * @return {Promise} promise resolving when the history is cleared
   */
  removeApplication(accountId) {
    return this.rpcRequest(accountId, { type: 'removeApplication' });
  }

  /**
   * MetaTrader trade response
   * @typedef {Object} MetatraderTradeResponse
   * @property {Number} numericCode numeric response code, see
   * https://www.mql5.com/en/docs/constants/errorswarnings/enum_trade_return_codes and
   * https://book.mql4.com/appendix/errors. Response codes which indicate success are 0, 10008-10010, 10025. The rest
   * codes are errors
   * @property {String} stringCode string response code, see
   * https://www.mql5.com/en/docs/constants/errorswarnings/enum_trade_return_codes and
   * https://book.mql4.com/appendix/errors. Response codes which indicate success are ERR_NO_ERROR,
   * TRADE_RETCODE_PLACED, TRADE_RETCODE_DONE, TRADE_RETCODE_DONE_PARTIAL, TRADE_RETCODE_NO_CHANGES. The rest codes are
   * errors.
   * @property {String} message human-readable response message
   * @property {String} orderId order id which was created/modified during the trade
   * @property {String} positionId position id which was modified during the trade
   */

  /**
   * Execute a trade on a connected MetaTrader account (see https://metaapi.cloud/docs/client/websocket/api/trade/).
   * @param {String} accountId id of the MetaTrader account to execute trade for
   * @param {MetatraderTrade} trade trade to execute (see docs for possible trade types)
   * @param {String} [application] application to use
   * @returns {Promise<MetatraderTradeResponse>} promise resolving with trade result
   * @throws {TradeError} on trade error, check error properties for error code details
   */
  async trade(accountId, trade, application) {
    let response = await this.rpcRequest(accountId, { type: 'trade', trade,
      application: application || this._application });
    response.response = response.response || {};
    response.response.stringCode = response.response.stringCode || response.response.description;
    response.response.numericCode = response.response.numericCode !== undefined ? response.response.numericCode : response.response.error;
    if (['ERR_NO_ERROR', 'TRADE_RETCODE_PLACED', 'TRADE_RETCODE_DONE', 'TRADE_RETCODE_DONE_PARTIAL', 'TRADE_RETCODE_NO_CHANGES'].includes(response.response.stringCode || response.response.description)) {
      return response.response;
    } else {
      throw new _tradeError2.default(response.response.message, response.response.numericCode, response.response.stringCode);
    }
  }

  /**
   * Creates a task that ensures the account gets subscribed to the server
   * @param {String} accountId account id to subscribe
   * @param {Number} [instanceNumber] instance index number
   */
  ensureSubscribe(accountId, instanceNumber) {
    this._subscriptionManager.scheduleSubscribe(accountId, instanceNumber);
  }

  /**
   * Subscribes to the Metatrader terminal events (see https://metaapi.cloud/docs/client/websocket/api/subscribe/).
   * @param {String} accountId id of the MetaTrader account to subscribe to
   * @param {Number} [instanceNumber] instance index number
   * @returns {Promise} promise which resolves when subscription started
   */
  subscribe(accountId, instanceNumber) {
    return this._subscriptionManager.subscribe(accountId, instanceNumber);
  }

  /**
   * Requests the terminal to start synchronization process
   * (see https://metaapi.cloud/docs/client/websocket/synchronizing/synchronize/).
   * @param {String} accountId id of the MetaTrader account to synchronize
   * @param {Number} instanceIndex instance index
   * @param {String} host name of host to synchronize with
   * @param {String} synchronizationId synchronization request id
   * @param {Date} startingHistoryOrderTime from what date to start synchronizing history orders from. If not specified,
   * the entire order history will be downloaded.
   * @param {Date} startingDealTime from what date to start deal synchronization from. If not specified, then all
   * history deals will be downloaded.
   * @param {String} specificationsMd5 specifications MD5 hash
   * @param {String} positionsMd5 positions MD5 hash
   * @param {String} ordersMd5 orders MD5 hash
   * @returns {Promise} promise which resolves when synchronization started
   */
  synchronize(accountId, instanceIndex, host, synchronizationId, startingHistoryOrderTime, startingDealTime, specificationsMd5, positionsMd5, ordersMd5) {
    const syncThrottler = this._socketInstances[this._socketInstancesByAccounts[accountId]].synchronizationThrottler;
    return syncThrottler.scheduleSynchronize(accountId, { requestId: synchronizationId,
      type: 'synchronize', startingHistoryOrderTime, startingDealTime, instanceIndex, host,
      specificationsMd5, positionsMd5, ordersMd5 });
  }

  /**
   * Waits for server-side terminal state synchronization to complete.
   * (see https://metaapi.cloud/docs/client/websocket/synchronizing/waitSynchronized/).
   * @param {String} accountId id of the MetaTrader account to synchronize
   * @param {Number} instanceNumber instance index number
   * @param {String} applicationPattern MetaApi application regular expression pattern, default is .*
   * @param {Number} timeoutInSeconds timeout in seconds, default is 300 seconds
   * @param {String} [application] application to synchronize with
   * @returns {Promise} promise which resolves when synchronization started
   */
  waitSynchronized(accountId, instanceNumber, applicationPattern, timeoutInSeconds, application) {
    return this.rpcRequest(accountId, { type: 'waitSynchronized', applicationPattern, timeoutInSeconds,
      instanceIndex: instanceNumber, application: application || this._application }, timeoutInSeconds + 1);
  }

  /**
   * Market data subscription
   * @typedef {Object} MarketDataSubscription
   * @property {string} type subscription type, one of quotes, candles, ticks, or marketDepth
   * @property {string} [timeframe] when subscription type is candles, defines the timeframe according to which the
   * candles must be generated. Allowed values for MT5 are 1m, 2m, 3m, 4m, 5m, 6m, 10m, 12m, 15m, 20m, 30m, 1h, 2h, 3h,
   * 4h, 6h, 8h, 12h, 1d, 1w, 1mn. Allowed values for MT4 are 1m, 5m, 15m 30m, 1h, 4h, 1d, 1w, 1mn
   * @property {number} [intervalInMilliseconds] defines how frequently the terminal will stream data to client. If not
   * set, then the value configured in account will be used
   */

  /**
   * Subscribes on market data of specified symbol (see
   * https://metaapi.cloud/docs/client/websocket/marketDataStreaming/subscribeToMarketData/).
   * @param {String} accountId id of the MetaTrader account
   * @param {Number} instanceNumber instance index number
   * @param {String} symbol symbol (e.g. currency pair or an index)
   * @param {Array<MarketDataSubscription>} subscriptions array of market data subscription to create or update
   * @returns {Promise} promise which resolves when subscription request was processed
   */
  subscribeToMarketData(accountId, instanceNumber, symbol, subscriptions) {
    return this.rpcRequest(accountId, { type: 'subscribeToMarketData', symbol, subscriptions,
      instanceIndex: instanceNumber });
  }

  /**
   * Refreshes market data subscriptions on the server to prevent them from expiring
   * @param {String} accountId id of the MetaTrader account
   * @param {Number} instanceNumber instance index number
   * @param {Array} subscriptions array of subscriptions to refresh
   */
  refreshMarketDataSubscriptions(accountId, instanceNumber, subscriptions) {
    return this.rpcRequest(accountId, { type: 'refreshMarketDataSubscriptions', subscriptions,
      instanceIndex: instanceNumber });
  }

  /**
   * Market data unsubscription
   * @typedef {Object} MarketDataUnsubscription
   * @property {string} type subscription type, one of quotes, candles, ticks, or marketDepth
   */

  /**
   * Unsubscribes from market data of specified symbol (see
   * https://metaapi.cloud/docs/client/websocket/marketDataStreaming/unsubscribeFromMarketData/).
   * @param {String} accountId id of the MetaTrader account
   * @param {Number} instanceNumber instance index
   * @param {String} symbol symbol (e.g. currency pair or an index)
   * @param {Array<MarketDataUnsubscription>} subscriptions array of subscriptions to cancel
   * @returns {Promise} promise which resolves when unsubscription request was processed
   */
  unsubscribeFromMarketData(accountId, instanceNumber, symbol, subscriptions) {
    return this.rpcRequest(accountId, { type: 'unsubscribeFromMarketData', symbol, subscriptions,
      instanceIndex: instanceNumber });
  }

  /**
   * Retrieves symbols available on an account (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveMarketData/readSymbols/).
   * @param {String} accountId id of the MetaTrader account to retrieve symbols for
   * @returns {Promise<Array<string>>} promise which resolves when symbols are retrieved
   */
  async getSymbols(accountId) {
    let response = await this.rpcRequest(accountId, { application: 'RPC', type: 'getSymbols' });
    return response.symbols;
  }

  /**
   * Retrieves specification for a symbol (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveMarketData/readSymbolSpecification/).
   * @param {String} accountId id of the MetaTrader account to retrieve symbol specification for
   * @param {String} symbol symbol to retrieve specification for
   * @returns {Promise<MetatraderSymbolSpecification>} promise which resolves when specification is retrieved
   */
  async getSymbolSpecification(accountId, symbol) {
    let response = await this.rpcRequest(accountId, { application: 'RPC', type: 'getSymbolSpecification', symbol });
    return response.specification;
  }

  /**
   * Retrieves price for a symbol (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveMarketData/readSymbolPrice/).
   * @param {String} accountId id of the MetaTrader account to retrieve symbol price for
   * @param {String} symbol symbol to retrieve price for
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderSymbolPrice>} promise which resolves when price is retrieved
   */
  async getSymbolPrice(accountId, symbol, keepSubscription = false) {
    let response = await this.rpcRequest(accountId, { application: 'RPC', type: 'getSymbolPrice', symbol,
      keepSubscription });
    return response.price;
  }

  /**
   * Retrieves price for a symbol (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveMarketData/readCandle/).
   * @param {string} accountId id of the MetaTrader account to retrieve candle for
   * @param {string} symbol symbol to retrieve candle for
   * @param {string} timeframe defines the timeframe according to which the candle must be generated. Allowed values for
   * MT5 are 1m, 2m, 3m, 4m, 5m, 6m, 10m, 12m, 15m, 20m, 30m, 1h, 2h, 3h, 4h, 6h, 8h, 12h, 1d, 1w, 1mn. Allowed values
   * for MT4 are 1m, 5m, 15m 30m, 1h, 4h, 1d, 1w, 1mn
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderCandle>} promise which resolves when candle is retrieved
   */
  async getCandle(accountId, symbol, timeframe, keepSubscription = false) {
    let response = await this.rpcRequest(accountId, { application: 'RPC', type: 'getCandle', symbol, timeframe,
      keepSubscription });
    return response.candle;
  }

  /**
   * Retrieves latest tick for a symbol. MT4 G1 accounts do not support this API (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveMarketData/readTick/).
   * @param {string} accountId id of the MetaTrader account to retrieve symbol tick for
   * @param {string} symbol symbol to retrieve tick for
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderTick>} promise which resolves when tick is retrieved
   */
  async getTick(accountId, symbol, keepSubscription = false) {
    let response = await this.rpcRequest(accountId, { application: 'RPC', type: 'getTick', symbol, keepSubscription });
    return response.tick;
  }

  /**
   * Retrieves latest order book for a symbol. MT4 accounts do not support this API (see
   * https://metaapi.cloud/docs/client/websocket/api/retrieveMarketData/readBook/).
   * @param {string} accountId id of the MetaTrader account to retrieve symbol order book for
   * @param {string} symbol symbol to retrieve order book for
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderBook>} promise which resolves when order book is retrieved
   */
  async getBook(accountId, symbol, keepSubscription = false) {
    let response = await this.rpcRequest(accountId, { application: 'RPC', type: 'getBook', symbol, keepSubscription });
    return response.book;
  }

  /**
   * Sends client uptime stats to the server.
   * @param {String} accountId id of the MetaTrader account to save uptime
   * @param {Object} uptime uptime statistics to send to the server
   * @returns {Promise} promise which resolves when uptime statistics is submitted
   */
  saveUptime(accountId, uptime) {
    return this.rpcRequest(accountId, { type: 'saveUptime', uptime });
  }

  /**
   * Unsubscribe from account (see
   * https://metaapi.cloud/docs/client/websocket/api/synchronizing/unsubscribe).
   * @param {String} accountId id of the MetaTrader account to unsubscribe
   * @returns {Promise} promise which resolves when socket unsubscribed
   */
  async unsubscribe(accountId) {
    try {
      await this._subscriptionManager.unsubscribe(accountId);
      delete this._socketInstancesByAccounts[accountId];
    } catch (err) {
      if (!['TimeoutError', 'NotFoundError'].includes(err.name)) {
        throw err;
      }
    }
  }

  /**
   * Adds synchronization listener for specific account
   * @param {String} accountId account id
   * @param {SynchronizationListener} listener synchronization listener to add
   */
  addSynchronizationListener(accountId, listener) {
    let listeners = this._synchronizationListeners[accountId];
    if (!listeners) {
      listeners = [];
      this._synchronizationListeners[accountId] = listeners;
    }
    listeners.push(listener);
  }

  /**
   * Removes synchronization listener for specific account
   * @param {String} accountId account id
   * @param {SynchronizationListener} listener synchronization listener to remove
   */
  removeSynchronizationListener(accountId, listener) {
    let listeners = this._synchronizationListeners[accountId];
    if (!listeners) {
      listeners = [];
    }
    listeners = listeners.filter(l => l !== listener);
    this._synchronizationListeners[accountId] = listeners;
  }

  /**
   * Adds latency listener
   * @param {LatencyListener} listener latency listener to add
   */
  addLatencyListener(listener) {
    this._latencyListeners.push(listener);
  }

  /**
   * Removes latency listener
   * @param {LatencyListener} listener latency listener to remove
   */
  removeLatencyListener(listener) {
    this._latencyListeners = this._latencyListeners.filter(l => l !== listener);
  }

  /**
   * Adds reconnect listener
   * @param {ReconnectListener} listener reconnect listener to add
   * @param {String} accountId account id of listener
   */
  addReconnectListener(listener, accountId) {
    this._reconnectListeners.push({ accountId, listener });
  }

  /**
   * Removes reconnect listener
   * @param {ReconnectListener} listener listener to remove
   */
  removeReconnectListener(listener) {
    this._reconnectListeners = this._reconnectListeners.filter(l => l.listener !== listener);
  }

  /**
   * Removes all listeners. Intended for use in unit tests.
   */
  removeAllListeners() {
    this._synchronizationListeners = {};
    this._reconnectListeners = [];
  }

  /**
   * Queues an account packet for processing
   * @param {Object} packet packet to process
   */
  queuePacket(packet) {
    const accountId = packet.accountId;
    const packets = this._packetOrderer.restoreOrder(packet).filter(p => p.type !== 'noop');
    if (this._sequentialEventProcessing && packet.sequenceNumber !== undefined) {
      const events = packets.map(packetItem => () => _promise2.default.resolve(this._processSynchronizationPacket(packetItem)));
      if (!this._eventQueues[accountId]) {
        this._eventQueues[accountId] = events;
        this._callAccountEvents(accountId);
      } else {
        this._eventQueues[accountId] = this._eventQueues[accountId].concat(events);
      }
    } else {
      packets.forEach(packetItem => this._processSynchronizationPacket(packetItem));
    }
  }

  /**
   * Queues account event for processing
   * @param {String} accountId account id
   * @param {Promise} event event to execute
   */
  queueEvent(accountId, event) {
    if (this._sequentialEventProcessing) {
      if (!this._eventQueues[accountId]) {
        this._eventQueues[accountId] = [event];
        this._callAccountEvents(accountId);
      } else {
        this._eventQueues[accountId].push(event);
      }
    } else {
      event();
    }
  }

  async _callAccountEvents(accountId) {
    if (this._eventQueues[accountId]) {
      while (this._eventQueues[accountId].length) {
        await this._eventQueues[accountId][0]();
        this._eventQueues[accountId].shift();
      }
      delete this._eventQueues[accountId];
    }
  }

  async _reconnect(socketInstanceIndex) {
    const instance = this.socketInstances[socketInstanceIndex];
    if (instance) {
      while (!instance.socket.connected && !instance.isReconnecting && instance.connected) {
        await this._tryReconnect(socketInstanceIndex);
      }
    }
  }

  _tryReconnect(socketInstanceIndex) {
    const instance = this.socketInstances[socketInstanceIndex];
    return new _promise2.default(resolve => setTimeout(async () => {
      if (!instance.socket.connected && !instance.isReconnecting && instance.connected) {
        try {
          instance.sessionId = _randomstring2.default.generate(32);
          const clientId = Math.random();
          instance.socket.close();
          instance.socket.io.opts.extraHeaders['Client-Id'] = clientId;
          instance.socket.io.opts.query.clientId = clientId;
          instance.isReconnecting = true;
          instance.socket.io.uri = await this._getServerUrl(socketInstanceIndex);
          instance.socket.connect();
        } catch (error) {
          instance.isReconnecting = false;
        }
      }
      resolve();
    }, 1000));
  }

  /**
   * Makes a RPC request
   * @param {String} accountId metatrader account id
   * @param {Object} request base request data
   * @param {Number} [timeoutInSeconds] request timeout in seconds
   */
  //eslint-disable-next-line complexity, max-statements
  async rpcRequest(accountId, request, timeoutInSeconds) {
    let socketInstanceIndex = null;
    if (this._socketInstancesByAccounts[accountId] !== undefined) {
      socketInstanceIndex = this._socketInstancesByAccounts[accountId];
    } else {
      while (this._subscribeLock && (new Date(this._subscribeLock.recommendedRetryTime).getTime() > Date.now() && this.subscribedAccountIds().length < this._subscribeLock.lockedAtAccounts || new Date(this._subscribeLock.lockedAtTime).getTime() + this._subscribeCooldownInSeconds * 1000 > Date.now() && this.subscribedAccountIds().length >= this._subscribeLock.lockedAtAccounts)) {
        await new _promise2.default(res => setTimeout(res, 1000));
      }
      for (let index = 0; index < this._socketInstances.length; index++) {
        const accountCounter = this.getAssignedAccounts(index).length;
        const instance = this.socketInstances[index];
        if (instance.subscribeLock) {
          if (instance.subscribeLock.type === 'LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_USER_PER_SERVER' && (new Date(instance.subscribeLock.recommendedRetryTime).getTime() > Date.now() || this.subscribedAccountIds(index).length >= instance.subscribeLock.lockedAtAccounts)) {
            continue;
          }
          if (instance.subscribeLock.type === 'LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_SERVER' && new Date(instance.subscribeLock.recommendedRetryTime).getTime() > Date.now() && this.subscribedAccountIds(index).length >= instance.subscribeLock.lockedAtAccounts) {
            continue;
          }
        }
        if (accountCounter < this._maxAccountsPerInstance) {
          socketInstanceIndex = index;
          break;
        }
      }
      if (socketInstanceIndex === null) {
        socketInstanceIndex = this._socketInstances.length;
        await this.connect();
      }
      this._socketInstancesByAccounts[accountId] = socketInstanceIndex;
    }
    const instance = this._socketInstances[socketInstanceIndex];
    if (!instance.connected) {
      await this.connect();
    } else if (!this.connected(socketInstanceIndex)) {
      await instance.connectResult;
    }
    if (request.type === 'subscribe') {
      request.sessionId = instance.sessionId;
    }
    if (['trade', 'subscribe'].includes(request.type)) {
      return this._makeRequest(accountId, request, timeoutInSeconds);
    }
    let retryCounter = 0;
    while (true) {
      //eslint-disable-line no-constant-condition
      try {
        return await this._makeRequest(accountId, request, timeoutInSeconds);
      } catch (err) {
        if (err.name === 'TooManyRequestsError') {
          let calcRetryCounter = retryCounter;
          let calcRequestTime = 0;
          while (calcRetryCounter < this._retries) {
            calcRetryCounter++;
            calcRequestTime += Math.min(Math.pow(2, calcRetryCounter) * this._minRetryDelayInSeconds, this._maxRetryDelayInSeconds) * 1000;
          }
          const retryTime = new Date(err.metadata.recommendedRetryTime).getTime();
          if (Date.now() + calcRequestTime > retryTime && retryCounter < this._retries) {
            if (Date.now() < retryTime) {
              await new _promise2.default(res => setTimeout(res, retryTime - Date.now()));
            }
            retryCounter++;
          } else {
            throw err;
          }
        } else if (['NotSynchronizedError', 'TimeoutError', 'NotAuthenticatedError', 'InternalError'].includes(err.name) && retryCounter < this._retries) {
          await new _promise2.default(res => setTimeout(res, Math.min(Math.pow(2, retryCounter) * this._minRetryDelayInSeconds, this._maxRetryDelayInSeconds) * 1000));
          retryCounter++;
        } else {
          throw err;
        }
        if (this._socketInstancesByAccounts[accountId] === undefined) {
          throw err;
        }
      }
    }
  }

  _makeRequest(accountId, request, timeoutInSeconds) {
    const socketInstance = this._socketInstances[this._socketInstancesByAccounts[accountId]];
    let requestId = request.requestId || _randomstring2.default.generate(32);
    request.timestamps = { clientProcessingStarted: new Date() };
    let result = _promise2.default.race([new _promise2.default((resolve, reject) => socketInstance.requestResolves[requestId] = { resolve, reject, type: request.type }), new _promise2.default((resolve, reject) => setTimeout(() => {
      reject(new _timeoutError2.default(`MetaApi websocket client request ${request.requestId} of type ${request.type} ` + 'timed out. Please make sure your account is connected to broker before retrying your request.'));
      delete socketInstance.requestResolves[requestId];
    }, timeoutInSeconds * 1000 || this._requestTimeout))]);
    request.accountId = accountId;
    request.application = request.application || this._application;
    if (!request.requestId) {
      request.requestId = requestId;
    }
    this._logger.debug(() => `${accountId}: Sending request: ${(0, _stringify2.default)(request)}`);
    socketInstance.socket.emit('request', request);
    return result;
  }

  // eslint-disable-next-line complexity
  _convertError(data) {
    if (data.error === 'ValidationError') {
      return new _errorHandler.ValidationError(data.message, data.details);
    } else if (data.error === 'NotFoundError') {
      return new _errorHandler.NotFoundError(data.message);
    } else if (data.error === 'NotSynchronizedError') {
      return new _notSynchronizedError2.default(data.message);
    } else if (data.error === 'TimeoutError') {
      return new _timeoutError2.default(data.message);
    } else if (data.error === 'NotAuthenticatedError') {
      return new _notConnectedError2.default(data.message);
    } else if (data.error === 'TradeError') {
      return new _tradeError2.default(data.message, data.numericCode, data.stringCode);
    } else if (data.error === 'UnauthorizedError') {
      this.close();
      return new _errorHandler.UnauthorizedError(data.message);
    } else if (data.error === 'TooManyRequestsError') {
      return new _errorHandler.TooManyRequestsError(data.message, data.metadata);
    } else {
      return new _errorHandler.InternalError(data.message);
    }
  }

  // eslint-disable-next-line complexity
  _convertIsoTimeToDate(packet) {
    // eslint-disable-next-line guard-for-in
    for (let field in packet) {
      let value = packet[field];
      if (typeof value === 'string' && field.match(/time$|Time$/) && !field.match(/brokerTime$|BrokerTime$|timeframe$/)) {
        packet[field] = new Date(value);
      }
      if (Array.isArray(value)) {
        for (let item of value) {
          this._convertIsoTimeToDate(item);
        }
      }
      if (typeof value === 'object') {
        this._convertIsoTimeToDate(value);
      }
    }
    if (packet && packet.timestamps) {
      // eslint-disable-next-line guard-for-in
      for (let field in packet.timestamps) {
        packet.timestamps[field] = new Date(packet.timestamps[field]);
      }
    }
    if (packet && packet.type === 'prices') {
      for (let price of packet.prices || []) {
        if (price.timestamps) {
          // eslint-disable-next-line guard-for-in
          for (let field in price.timestamps) {
            price.timestamps[field] = new Date(price.timestamps[field]);
          }
        }
      }
    }
  }

  /**
   * MetaTrader symbol specification. Contains symbol specification (see
   * https://metaapi.cloud/docs/client/models/metatraderSymbolSpecification/)
   * @typedef {Object} MetatraderSymbolSpecification
   * @property {String} symbol symbol (e.g. a currency pair or an index)
   * @property {Number} tickSize tick size
   * @property {Number} minVolume minimum order volume for the symbol
   * @property {Number} maxVolume maximum order volume for the symbol
   * @property {Number} volumeStep order volume step for the symbol
   * @property {Array<String>} list of allowed order filling modes. Can contain ORDER_FILLING_FOK, ORDER_FILLING_IOC or
   * both. See https://www.mql5.com/en/docs/constants/environment_state/marketinfoconstants#symbol_filling_mode for more
   * details.
   * @property {String} deal execution mode. Possible values are SYMBOL_TRADE_EXECUTION_REQUEST,
   * SYMBOL_TRADE_EXECUTION_INSTANT, SYMBOL_TRADE_EXECUTION_MARKET, SYMBOL_TRADE_EXECUTION_EXCHANGE. See
   * https://www.mql5.com/en/docs/constants/environment_state/marketinfoconstants#enum_symbol_trade_execution for more
   * details.
   * @property {Number} contractSize trade contract size
   * @property {MetatraderSessions} quoteSessions quote sessions, indexed by day of week
   * @property {MetatraderSessions} tradeSessions trade sessions, indexed by day of week
   * @property {String} [tradeMode] order execution type. Possible values are SYMBOL_TRADE_MODE_DISABLED,
   * SYMBOL_TRADE_MODE_LONGONLY, SYMBOL_TRADE_MODE_SHORTONLY, SYMBOL_TRADE_MODE_CLOSEONLY, SYMBOL_TRADE_MODE_FULL. See
   * https://www.mql5.com/en/docs/constants/environment_state/marketinfoconstants#enum_symbol_trade_mode for more
   * details
   * @property {Number} [bondAccruedInterest] accrued interest – accumulated coupon interest, i.e. part of the coupon
   * interest calculated in proportion to the number of days since the coupon bond issuance or the last coupon interest
   * payment
   * @property {Number} [bondFaceValue] face value – initial bond value set by the issuer
   * @property {Number} [optionStrike] the strike price of an option. The price at which an option buyer can buy (in a
   * Call option) or sell (in a Put option) the underlying asset, and the option seller is obliged to sell or buy the
   * appropriate amount of the underlying asset.
   * @property {Number} [optionPriceSensivity] option/warrant sensitivity shows by how many points the price of the
   * option's underlying asset should change so that the price of the option changes by one point
   * @property {Number} [liquidityRate] liquidity Rate is the share of the asset that can be used for the margin
   * @property {Number} initialMargin initial margin means the amount in the margin currency required for opening a
   * position with the volume of one lot. It is used for checking a client's assets when he or she enters the market
   * @property {Number} maintenanceMargin the maintenance margin. If it is set, it sets the margin amount in the margin
   * currency of the symbol, charged from one lot. It is used for checking a client's assets when his/her account state
   * changes. If the maintenance margin is equal to 0, the initial margin is used
   * @property {Number} hedgedMargin contract size or margin value per one lot of hedged positions (oppositely directed
   * positions of one symbol). Two margin calculation methods are possible for hedged positions. The calculation method
   * is defined by the broker
   * @property {Boolean} [hedgedMarginUsesLargerLeg] calculating hedging margin using the larger leg (Buy or Sell)
   * @property {String} marginCurrency margin currency
   * @property {String} priceCalculationMode contract price calculation mode. One of SYMBOL_CALC_MODE_UNKNOWN,
   * SYMBOL_CALC_MODE_FOREX, SYMBOL_CALC_MODE_FOREX_NO_LEVERAGE, SYMBOL_CALC_MODE_FUTURES, SYMBOL_CALC_MODE_CFD,
   * SYMBOL_CALC_MODE_CFDINDEX, SYMBOL_CALC_MODE_CFDLEVERAGE, SYMBOL_CALC_MODE_EXCH_STOCKS,
   * SYMBOL_CALC_MODE_EXCH_FUTURES, SYMBOL_CALC_MODE_EXCH_FUTURES_FORTS, SYMBOL_CALC_MODE_EXCH_BONDS,
   * SYMBOL_CALC_MODE_EXCH_STOCKS_MOEX, SYMBOL_CALC_MODE_EXCH_BONDS_MOEX, SYMBOL_CALC_MODE_SERV_COLLATERAL. See
   * https://www.mql5.com/en/docs/constants/environment_state/marketinfoconstants#enum_symbol_calc_mode for more details
   * @property {String} baseCurrency base currency
   * @property {String} [profitCurrency] profit currency
   * @property {String} swapMode swap calculation model. Allowed values are SYMBOL_SWAP_MODE_DISABLED,
   * SYMBOL_SWAP_MODE_POINTS, SYMBOL_SWAP_MODE_CURRENCY_SYMBOL, SYMBOL_SWAP_MODE_CURRENCY_MARGIN,
   * SYMBOL_SWAP_MODE_CURRENCY_DEPOSIT, SYMBOL_SWAP_MODE_INTEREST_CURRENT, SYMBOL_SWAP_MODE_INTEREST_OPEN,
   * SYMBOL_SWAP_MODE_REOPEN_CURRENT, SYMBOL_SWAP_MODE_REOPEN_BID. See
   * https://www.mql5.com/en/docs/constants/environment_state/marketinfoconstants#enum_symbol_swap_mode for more details
   * @property {Number} [swapLong] long swap value
   * @property {Number} [swapShort] short swap value
   * @property {String} [swapRollover3Days] day of week to charge 3 days swap rollover. Allowed values are SUNDAY,
   * MONDAY, TUESDAY, WEDNESDAY, THURDAY, FRIDAY, SATURDAY, NONE
   * @property {Array<String>} allowedExpirationModes allowed order expiration modes. Allowed values are
   * SYMBOL_EXPIRATION_GTC, SYMBOL_EXPIRATION_DAY, SYMBOL_EXPIRATION_SPECIFIED, SYMBOL_EXPIRATION_SPECIFIED_DAY.
   * See https://www.mql5.com/en/docs/constants/environment_state/marketinfoconstants#symbol_expiration_mode for more
   * details
   * @property {Array<String>} allowedOrderTypes allowed order types. Allowed values are SYMBOL_ORDER_MARKET,
   * SYMBOL_ORDER_LIMIT, SYMBOL_ORDER_STOP, SYMBOL_ORDER_STOP_LIMIT, SYMBOL_ORDER_SL, SYMBOL_ORDER_TP,
   * SYMBOL_ORDER_CLOSEBY. See
   * https://www.mql5.com/en/docs/constants/environment_state/marketinfoconstants#symbol_order_mode for more details
   * @property {String} orderGTCMode if the expirationMode property is set to SYMBOL_EXPIRATION_GTC (good till
   * canceled), the expiration of pending orders, as well as of Stop Loss/Take Profit orders should be additionally set
   * using this enumeration. Allowed values are SYMBOL_ORDERS_GTC, SYMBOL_ORDERS_DAILY,
   * SYMBOL_ORDERS_DAILY_EXCLUDING_STOPS. See
   * https://www.mql5.com/en/docs/constants/environment_state/marketinfoconstants#enum_symbol_order_gtc_mode for more
   * details
   * @property {Number} digits digits after a decimal point
   * @property {Number} point point size
   * @property {String} [path] path in the symbol tree
   * @property {String} description symbol description
   * @property {Date} [startTime] date of the symbol trade beginning (usually used for futures)
   * @property {Date} [expirationTime] date of the symbol trade end (usually used for futures)
   */

  /**
   * Metatrader trade or quote session container, indexed by weekday
   * @typedef {Object} MetatraderSessions
   * @property {Array<MetatraderSession>} [SUNDAY] array of sessions for SUNDAY
   * @property {Array<MetatraderSession>} [MONDAY] array of sessions for MONDAY
   * @property {Array<MetatraderSession>} [TUESDAY] array of sessions for TUESDAY
   * @property {Array<MetatraderSession>} [WEDNESDAY] array of sessions for WEDNESDAY
   * @property {Array<MetatraderSession>} [THURSDAY] array of sessions for THURSDAY
   * @property {Array<MetatraderSession>} [FRIDAY] array of sessions for FRIDAY
   * @property {Array<MetatraderSession>} [SATURDAY] array of sessions for SATURDAY
   */

  /**
   * Metatrader trade or quote session
   * @typedef {Object} MetatraderSession
   * @property {String} from session start time, in hh.mm.ss.SSS format
   * @property {String} to session end time, in hh.mm.ss.SSS format
   */

  /**
   * MetaTrader symbol price. Contains current price for a symbol (see
   * https://metaapi.cloud/docs/client/models/metatraderSymbolPrice/)
   * @typedef {Object} MetatraderSymbolPrice
   * @property {String} symbol symbol (e.g. a currency pair or an index)
   * @property {Number} bid bid price
   * @property {Number} ask ask price
   * @property {Number} profitTickValue tick value for a profitable position
   * @property {Number} lossTickValue tick value for a losing position
   * @property {Number} [accountCurrencyExchangeRate] current exchange rate of account currency into account base
   * currency (USD if you did not override it)
   * @property {Date} time quote time, in ISO format
   * @property {String} brokerTime time quote time, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   */

  /**
   * MetaTrader candle
   * @typedef {Object} MetatraderCandle
   * @property {string} symbol symbol (e.g. currency pair or an index)
   * @property {string} timeframe timeframe candle was generated for, e.g. 1h. One of 1m, 2m, 3m, 4m, 5m, 6m, 10m, 12m,
   * 15m, 20m, 30m, 1h, 2h, 3h, 4h, 6h, 8h, 12h, 1d, 1w, 1mn
   * @property {Date} time candle opening time
   * @property {string} brokerTime candle opening time, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {number} open open price
   * @property {number} high high price
   * @property {number} low low price
   * @property {number} close close price
   * @property {number} tickVolume tick volume, i.e. number of ticks inside the candle
   * @property {number} spread spread in points
   * @property {number} volume trade volume
   */

  /**
   * MetaTrader tick data
   * @typedef {Object} MetatraderTick
   * @property {string} symbol symbol (e.g. a currency pair or an index)
   * @property {Date} time time
   * @property {string} brokerTime time, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {number} [bid] bid price
   * @property {number} [ask] ask price
   * @property {number} [last] last deal price
   * @property {number} [volume] volume for the current last deal price
   * @property {string} side is tick a result of buy or sell deal, one of buy or sell
   */

  /**
   * MetaTrader order book
   * @typedef {Object} MetatraderBook
   * @property {string} symbol symbol (e.g. a currency pair or an index)
   * @property {Date} time time
   * @property {string} brokerTime time, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {Array<MetatraderBookEntry>} book list of order book entries
   */

  /**
   * MetaTrader order book entry
   * @typedef {Object} MetatraderBookEntry
   * @property {string} type entry type, one of BOOK_TYPE_SELL, BOOK_TYPE_BUY, BOOK_TYPE_SELL_MARKET,
   * BOOK_TYPE_BUY_MARKET
   * @property {number} price price
   * @property {number} volume volume
   */

  // eslint-disable-next-line complexity,max-statements
  async _processSynchronizationPacket(data) {
    try {
      const socketInstance = this._socketInstances[this._socketInstancesByAccounts[data.accountId]];
      if (data.synchronizationId && socketInstance) {
        socketInstance.synchronizationThrottler.updateSynchronizationId(data.synchronizationId);
      }
      const instanceNumber = data.instanceIndex || 0;
      let instanceId = data.accountId + ':' + instanceNumber + ':' + (data.host || 0);
      let instanceIndex = instanceNumber + ':' + (data.host || 0);

      const _processEvent = async (event, eventName) => {
        const startTime = Date.now();
        let isLongEvent = false;
        let isEventDone = false;

        const checkLongEvent = async () => {
          await new _promise2.default(res => setTimeout(res, 1000));
          if (!isEventDone) {
            isLongEvent = true;
            this._logger.warn(`${instanceId}: event ${eventName} is taking more than 1 second to process`);
          }
        };

        checkLongEvent();
        await event;
        isEventDone = true;
        if (isLongEvent) {
          this._logger.warn(`${instanceId}: event ${eventName} finished in ` + `${Math.floor((Date.now() - startTime) / 1000)} seconds`);
        }
      };

      const isOnlyActiveInstance = () => {
        const activeInstanceIds = (0, _keys2.default)(this._connectedHosts).filter(instance => instance.startsWith(data.accountId + ':' + instanceNumber));
        return !activeInstanceIds.length || activeInstanceIds.length === 1 && activeInstanceIds[0] === instanceId;
      };

      const cancelDisconnectTimer = () => {
        if (this._statusTimers[instanceId]) {
          clearTimeout(this._statusTimers[instanceId]);
        }
      };

      const resetDisconnectTimer = () => {
        cancelDisconnectTimer();
        this._statusTimers[instanceId] = setTimeout(() => {
          if (isOnlyActiveInstance()) {
            this._subscriptionManager.onTimeout(data.accountId, instanceNumber);
          }
          this.queueEvent(data.accountId, () => _promise2.default.resolve(onDisconnected(true)));
          clearTimeout(this._statusTimers[instanceId]);
        }, 60000);
      };

      // eslint-disable-next-line complexity
      const onDisconnected = async (isTimeout = false) => {
        if (this._connectedHosts[instanceId]) {
          if (isOnlyActiveInstance()) {
            const onDisconnectedPromises = [];
            if (!isTimeout) {
              onDisconnectedPromises.push(this._subscriptionManager.onDisconnected(data.accountId, instanceNumber));
            }
            for (let listener of this._synchronizationListeners[data.accountId] || []) {
              onDisconnectedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onDisconnected(instanceIndex)), 'onDisconnected'))
              // eslint-disable-next-line no-console
              .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about disconnected event', err)));
            }
            await _promise2.default.all(onDisconnectedPromises);
          } else {
            const onStreamClosedPromises = [];
            this._packetOrderer.onStreamClosed(instanceId);
            if (socketInstance) {
              socketInstance.synchronizationThrottler.removeIdByParameters(data.accountId, instanceNumber, data.host);
            }
            for (let listener of this._synchronizationListeners[data.accountId] || []) {
              onStreamClosedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onStreamClosed(instanceIndex)), 'onStreamClosed'))
              // eslint-disable-next-line no-console
              .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about stream closed event', err)));
            }
            await _promise2.default.all(onStreamClosedPromises);
          }
          delete this._connectedHosts[instanceId];
        }
      };
      if (data.type === 'authenticated') {
        resetDisconnectTimer();
        if (!data.sessionId || socketInstance && data.sessionId === socketInstance.sessionId) {
          this._connectedHosts[instanceId] = data.host;
          const onConnectedPromises = [];
          for (let listener of this._synchronizationListeners[data.accountId] || []) {
            onConnectedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onConnected(instanceIndex, data.replicas)), 'onConnected'))
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about connected event', err)));
          }
          this._subscriptionManager.cancelSubscribe(data.accountId + ':' + instanceNumber);
          await _promise2.default.all(onConnectedPromises);
        }
      } else if (data.type === 'disconnected') {
        cancelDisconnectTimer();
        await onDisconnected();
      } else if (data.type === 'synchronizationStarted') {
        const promises = [];
        this._synchronizationFlags[data.synchronizationId] = {
          accountId: data.accountId,
          positionsUpdated: data.positionsUpdated !== undefined ? data.positionsUpdated : true,
          ordersUpdated: data.ordersUpdated !== undefined ? data.ordersUpdated : true
        };
        for (let listener of this._synchronizationListeners[data.accountId] || []) {
          promises.push(_promise2.default.resolve((async () => {
            await _processEvent(_promise2.default.resolve(listener.onSynchronizationStarted(instanceIndex, data.specificationsUpdated !== undefined ? data.specificationsUpdated : true, data.positionsUpdated !== undefined ? data.positionsUpdated : true, data.ordersUpdated !== undefined ? data.ordersUpdated : true)), 'onSynchronizationStarted');
          })())
          // eslint-disable-next-line no-console
          .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about synchronization started event', err)));
        }
        await _promise2.default.all(promises);
      } else if (data.type === 'accountInformation') {
        if (data.accountInformation) {
          const onAccountInformationUpdatedPromises = [];
          for (let listener of this._synchronizationListeners[data.accountId] || []) {
            onAccountInformationUpdatedPromises.push(_promise2.default.resolve((async () => {
              await _processEvent(_promise2.default.resolve(listener.onAccountInformationUpdated(instanceIndex, data.accountInformation)), 'onAccountInformationUpdated');
              if (this._synchronizationFlags[data.synchronizationId] && !this._synchronizationFlags[data.synchronizationId].positionsUpdated) {
                await _processEvent(_promise2.default.resolve(listener.onPositionsSynchronized(instanceIndex, data.synchronizationId)), 'onPositionsSynchronized');
                if (!this._synchronizationFlags[data.synchronizationId].ordersUpdated) {
                  await _processEvent(_promise2.default.resolve(listener.onPendingOrdersSynchronized(instanceIndex, data.synchronizationId)), 'onPendingOrdersSynchronized');
                }
              }
            })())
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about accountInformation event', err)));
          }
          await _promise2.default.all(onAccountInformationUpdatedPromises);
          if (this._synchronizationFlags[data.synchronizationId] && !this._synchronizationFlags[data.synchronizationId].positionsUpdated && !this._synchronizationFlags[data.synchronizationId].ordersUpdated) {
            delete this._synchronizationFlags[data.synchronizationId];
          }
        }
      } else if (data.type === 'deals') {
        for (let deal of data.deals || []) {
          const onDealAddedPromises = [];
          for (let listener of this._synchronizationListeners[data.accountId] || []) {
            onDealAddedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onDealAdded(instanceIndex, deal)), 'onDealAdded'))
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about deals event', err)));
          }
          await _promise2.default.all(onDealAddedPromises);
        }
      } else if (data.type === 'orders') {
        const onPendingOrdersReplacedPromises = [];
        for (let listener of this._synchronizationListeners[data.accountId] || []) {
          onPendingOrdersReplacedPromises.push(_promise2.default.resolve((async () => {
            await _processEvent(_promise2.default.resolve(listener.onPendingOrdersReplaced(instanceIndex, data.orders || [])), 'onPendingOrdersReplaced');
            await _processEvent(_promise2.default.resolve(listener.onPendingOrdersSynchronized(instanceIndex, data.synchronizationId)), 'onPendingOrdersSynchronized');
          })())
          // eslint-disable-next-line no-console
          .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about orders event', err)));
        }
        await _promise2.default.all(onPendingOrdersReplacedPromises);
        if (this._synchronizationFlags[data.synchronizationId]) {
          delete this._synchronizationFlags[data.synchronizationId];
        }
      } else if (data.type === 'historyOrders') {
        for (let historyOrder of data.historyOrders || []) {
          const onHistoryOrderAddedPromises = [];
          for (let listener of this._synchronizationListeners[data.accountId] || []) {
            onHistoryOrderAddedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onHistoryOrderAdded(instanceIndex, historyOrder)), 'onHistoryOrderAdded'))
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about historyOrders event', err)));
          }
          await _promise2.default.all(onHistoryOrderAddedPromises);
        }
      } else if (data.type === 'positions') {
        const onPositionsReplacedPromises = [];
        for (let listener of this._synchronizationListeners[data.accountId] || []) {
          onPositionsReplacedPromises.push(_promise2.default.resolve((async () => {
            await _processEvent(_promise2.default.resolve(listener.onPositionsReplaced(instanceIndex, data.positions || [])), 'onPositionsReplaced');
            await _processEvent(_promise2.default.resolve(listener.onPositionsSynchronized(instanceIndex, data.synchronizationId)), 'onPositionsSynchronized');
            if (this._synchronizationFlags[data.synchronizationId] && !this._synchronizationFlags[data.synchronizationId].ordersUpdated) {
              await _processEvent(_promise2.default.resolve(listener.onPendingOrdersSynchronized(instanceIndex, data.synchronizationId)), 'onPendingOrdersSynchronized');
            }
          })())
          // eslint-disable-next-line no-console
          .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about positions event', err)));
        }
        await _promise2.default.all(onPositionsReplacedPromises);
        if (this._synchronizationFlags[data.synchronizationId] && !this._synchronizationFlags[data.synchronizationId].ordersUpdated) {
          delete this._synchronizationFlags[data.synchronizationId];
        }
      } else if (data.type === 'update') {
        if (data.accountInformation) {
          const onAccountInformationUpdatedPromises = [];
          for (let listener of this._synchronizationListeners[data.accountId] || []) {
            onAccountInformationUpdatedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onAccountInformationUpdated(instanceIndex, data.accountInformation)), 'onAccountInformationUpdated'))
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about update event', err)));
          }
          await _promise2.default.all(onAccountInformationUpdatedPromises);
        }
        for (let position of data.updatedPositions || []) {
          const onPositionUpdatedPromises = [];
          for (let listener of this._synchronizationListeners[data.accountId] || []) {
            onPositionUpdatedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onPositionUpdated(instanceIndex, position)), 'onPositionUpdated'))
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about update event', err)));
          }
          await _promise2.default.all(onPositionUpdatedPromises);
        }
        for (let positionId of data.removedPositionIds || []) {
          const onPositionRemovedPromises = [];
          for (let listener of this._synchronizationListeners[data.accountId] || []) {
            onPositionRemovedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onPositionRemoved(instanceIndex, positionId)), 'onPositionRemoved'))
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about update event', err)));
          }
          await _promise2.default.all(onPositionRemovedPromises);
        }
        for (let order of data.updatedOrders || []) {
          const onPendingOrderUpdatedPromises = [];
          for (let listener of this._synchronizationListeners[data.accountId] || []) {
            onPendingOrderUpdatedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onPendingOrderUpdated(instanceIndex, order)), 'onPendingOrderUpdated'))
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about update event', err)));
          }
          await _promise2.default.all(onPendingOrderUpdatedPromises);
        }
        for (let orderId of data.completedOrderIds || []) {
          const onPendingOrderCompletedPromises = [];
          for (let listener of this._synchronizationListeners[data.accountId] || []) {
            onPendingOrderCompletedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onPendingOrderCompleted(instanceIndex, orderId)), 'onPendingOrderCompleted'))
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about update event', err)));
          }
          await _promise2.default.all(onPendingOrderCompletedPromises);
        }
        for (let historyOrder of data.historyOrders || []) {
          const onHistoryOrderAddedPromises = [];
          for (let listener of this._synchronizationListeners[data.accountId] || []) {
            onHistoryOrderAddedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onHistoryOrderAdded(instanceIndex, historyOrder)), 'onHistoryOrderAdded'))
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about update event', err)));
          }
          await _promise2.default.all(onHistoryOrderAddedPromises);
        }
        for (let deal of data.deals || []) {
          const onDealAddedPromises = [];
          for (let listener of this._synchronizationListeners[data.accountId] || []) {
            onDealAddedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onDealAdded(instanceIndex, deal)), 'onDealAdded'))
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about update event', err)));
          }
          await _promise2.default.all(onDealAddedPromises);
        }
        if (data.timestamps) {
          data.timestamps.clientProcessingFinished = new Date();
          const onUpdatePromises = [];
          // eslint-disable-next-line max-depth
          for (let listener of this._latencyListeners || []) {
            onUpdatePromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onUpdate(data.accountId, data.timestamps)), 'onUpdate'))
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify latency ` + 'listener about update event', err)));
          }
          await _promise2.default.all(onUpdatePromises);
        }
      } else if (data.type === 'dealSynchronizationFinished') {
        const onDealsSynchronizedPromises = [];
        for (let listener of this._synchronizationListeners[data.accountId] || []) {
          if (socketInstance) {
            socketInstance.synchronizationThrottler.removeSynchronizationId(data.synchronizationId);
          }
          onDealsSynchronizedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onDealsSynchronized(instanceIndex, data.synchronizationId)), 'onDealsSynchronized'))
          // eslint-disable-next-line no-console
          .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener about ` + 'dealSynchronizationFinished event', err)));
        }
        await _promise2.default.all(onDealsSynchronizedPromises);
      } else if (data.type === 'orderSynchronizationFinished') {
        const onHistoryOrdersSynchronizedPromises = [];
        for (let listener of this._synchronizationListeners[data.accountId] || []) {
          onHistoryOrdersSynchronizedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onHistoryOrdersSynchronized(instanceIndex, data.synchronizationId)), 'onHistoryOrdersSynchronized'))
          // eslint-disable-next-line no-console
          .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener about ` + 'orderSynchronizationFinished event', err)));
        }
        await _promise2.default.all(onHistoryOrdersSynchronizedPromises);
      } else if (data.type === 'status') {
        if (!this._connectedHosts[instanceId]) {
          if (this._statusTimers[instanceId] && data.authenticated && (this._subscriptionManager.isDisconnectedRetryMode(data.accountId, instanceNumber) || !this._subscriptionManager.isAccountSubscribing(data.accountId, instanceNumber))) {
            this._subscriptionManager.cancelSubscribe(data.accountId + ':' + instanceNumber);
            await new _promise2.default(res => setTimeout(res, 10));
            // eslint-disable-next-line no-console
            this._logger.info('it seems like we are not connected to a running API ' + 'server yet, retrying subscription for account ' + instanceId);
            this.ensureSubscribe(data.accountId, instanceNumber);
          }
        } else {
          resetDisconnectTimer();
          const onBrokerConnectionStatusChangedPromises = [];
          for (let listener of this._synchronizationListeners[data.accountId] || []) {
            onBrokerConnectionStatusChangedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onBrokerConnectionStatusChanged(instanceIndex, !!data.connected)), 'onBrokerConnectionStatusChanged'))
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify ` + 'listener about brokerConnectionStatusChanged event', err)));
          }
          await _promise2.default.all(onBrokerConnectionStatusChangedPromises);
          if (data.healthStatus) {
            const onHealthStatusPromises = [];
            // eslint-disable-next-line max-depth
            for (let listener of this._synchronizationListeners[data.accountId] || []) {
              onHealthStatusPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onHealthStatus(instanceIndex, data.healthStatus)), 'onHealthStatus'))
              // eslint-disable-next-line no-console
              .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify ` + 'listener about server-side healthStatus event', err)));
            }
            await _promise2.default.all(onHealthStatusPromises);
          }
        }
      } else if (data.type === 'downgradeSubscription') {
        // eslint-disable-next-line no-console
        this._logger.info(`${data.accountId}:${instanceIndex}: Market data subscriptions for symbol ` + `${data.symbol} were downgraded by the server due to rate limits. Updated subscriptions: ` + `${(0, _stringify2.default)(data.updates)}, removed subscriptions: ${(0, _stringify2.default)(data.unsubscriptions)}. ` + 'Please read https://metaapi.cloud/docs/client/rateLimiting/ for more details.');
        const onSubscriptionDowngradePromises = [];
        for (let listener of this._synchronizationListeners[data.accountId] || []) {
          onSubscriptionDowngradePromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onSubscriptionDowngraded(instanceIndex, data.symbol, data.updates, data.unsubscriptions)), 'onSubscriptionDowngraded'))
          // eslint-disable-next-line no-console
          .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about subscription downgrade event', err)));
        }
        await _promise2.default.all(onSubscriptionDowngradePromises);
      } else if (data.type === 'specifications') {
        const onSymbolSpecificationsUpdatedPromises = [];
        for (let listener of this._synchronizationListeners[data.accountId] || []) {
          onSymbolSpecificationsUpdatedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onSymbolSpecificationsUpdated(instanceIndex, data.specifications || [], data.removedSymbols || [])), 'onSymbolSpecificationsUpdated'))
          // eslint-disable-next-line no-console
          .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about specifications updated event', err)));
        }
        await _promise2.default.all(onSymbolSpecificationsUpdatedPromises);
        for (let specification of data.specifications || []) {
          const onSymbolSpecificationUpdatedPromises = [];
          for (let listener of this._synchronizationListeners[data.accountId] || []) {
            onSymbolSpecificationUpdatedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onSymbolSpecificationUpdated(instanceIndex, specification)), 'onSymbolSpecificationUpdated'))
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about specification updated event', err)));
          }
          await _promise2.default.all(onSymbolSpecificationUpdatedPromises);
        }
        for (let removedSymbol of data.removedSymbols || []) {
          const onSymbolSpecificationRemovedPromises = [];
          for (let listener of this._synchronizationListeners[data.accountId] || []) {
            onSymbolSpecificationRemovedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onSymbolSpecificationRemoved(instanceIndex, removedSymbol)), 'onSymbolSpecificationRemoved'))
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about specifications removed event', err)));
          }
          await _promise2.default.all(onSymbolSpecificationRemovedPromises);
        }
      } else if (data.type === 'prices') {
        let prices = data.prices || [];
        let candles = data.candles || [];
        let ticks = data.ticks || [];
        let books = data.books || [];
        const onSymbolPricesUpdatedPromises = [];
        for (let listener of this._synchronizationListeners[data.accountId] || []) {
          if (prices.length) {
            onSymbolPricesUpdatedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onSymbolPricesUpdated(instanceIndex, prices, data.equity, data.margin, data.freeMargin, data.marginLevel, data.accountCurrencyExchangeRate)), 'onSymbolPricesUpdated'))
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about prices event', err)));
          }
          if (candles.length) {
            onSymbolPricesUpdatedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onCandlesUpdated(instanceIndex, candles, data.equity, data.margin, data.freeMargin, data.marginLevel, data.accountCurrencyExchangeRate)), 'onCandlesUpdated'))
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about candles event', err)));
          }
          if (ticks.length) {
            onSymbolPricesUpdatedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onTicksUpdated(instanceIndex, ticks, data.equity, data.margin, data.freeMargin, data.marginLevel, data.accountCurrencyExchangeRate)), 'onTicksUpdated'))
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about ticks event', err)));
          }
          if (books.length) {
            onSymbolPricesUpdatedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onBooksUpdated(instanceIndex, books, data.equity, data.margin, data.freeMargin, data.marginLevel, data.accountCurrencyExchangeRate)), 'onBooksUpdated'))
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about books event', err)));
          }
        }
        await _promise2.default.all(onSymbolPricesUpdatedPromises);
        for (let price of prices) {
          const onSymbolPriceUpdatedPromises = [];
          for (let listener of this._synchronizationListeners[data.accountId] || []) {
            onSymbolPriceUpdatedPromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onSymbolPriceUpdated(instanceIndex, price)), 'onSymbolPriceUpdated'))
            // eslint-disable-next-line no-console
            .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify listener ` + 'about price event', err)));
          }
          await _promise2.default.all(onSymbolPriceUpdatedPromises);
        }
        for (let price of prices) {
          if (price.timestamps) {
            price.timestamps.clientProcessingFinished = new Date();
            const onSymbolPricePromises = [];
            // eslint-disable-next-line max-depth
            for (let listener of this._latencyListeners || []) {
              onSymbolPricePromises.push(_promise2.default.resolve(_processEvent(_promise2.default.resolve(listener.onSymbolPrice(data.accountId, price.symbol, price.timestamps)), 'onSymbolPrice'))
              // eslint-disable-next-line no-console
              .catch(err => this._logger.error(`${data.accountId}:${instanceIndex}: Failed to notify latency ` + 'listener about price event', err)));
            }
            await _promise2.default.all(onSymbolPricePromises);
          }
        }
      }
    } catch (err) {
      // eslint-disable-next-line no-console
      this._logger.error('Failed to process incoming synchronization packet', err);
    }
  }

  async _fireReconnected(socketInstanceIndex) {
    try {
      const reconnectListeners = [];
      for (let listener of this._reconnectListeners) {
        if (this._socketInstancesByAccounts[listener.accountId] === socketInstanceIndex) {
          reconnectListeners.push(listener);
        }
      }
      (0, _keys2.default)(this._synchronizationFlags).forEach(synchronizationId => {
        if (this._socketInstancesByAccounts[this._synchronizationFlags[synchronizationId].accountId] === socketInstanceIndex) {
          delete this._synchronizationFlags[synchronizationId];
        }
      });
      const reconnectAccountIds = reconnectListeners.map(listener => listener.accountId);
      this._subscriptionManager.onReconnected(socketInstanceIndex, reconnectAccountIds);
      this._packetOrderer.onReconnected(reconnectAccountIds);

      for (let listener of reconnectListeners) {
        _promise2.default.resolve(listener.listener.onReconnected()).catch(err => this._logger.error('Failed to notify reconnect listener', err));
      }
    } catch (err) {
      this._logger.error('Failed to process reconnected event', err);
    }
  }

  // eslint-disable-next-line complexity
  async _getServerUrl(socketInstanceIndex) {
    while (this.socketInstances[socketInstanceIndex].connected) {
      try {
        let isDefaultRegion = !this._region;
        if (this._region) {
          const opts = {
            url: `https://mt-provisioning-api-v1.${this._domain}/users/current/regions`,
            method: 'GET',
            headers: {
              'auth-token': this._token
            },
            json: true
          };
          const regions = await this._httpClient.request(opts);
          if (!regions.includes(this._region)) {
            const errorMessage = `The region "${this._region}" you are trying to connect to does not exist ` + 'or is not available to you. Please specify a correct region name in the ' + 'region MetaApi constructor option.';
            throw new _errorHandler.NotFoundError(errorMessage);
          }
          if (this._region === regions[0]) {
            isDefaultRegion = true;
          }
        }

        let url;
        if (this._useSharedClientApi) {
          if (isDefaultRegion) {
            url = this._url;
          } else {
            url = `https://${this._hostname}.${this._region}.${this._domain}`;
          }
        } else {
          const opts = {
            url: `https://mt-provisioning-api-v1.${this._domain}/users/current/servers/mt-client-api`,
            method: 'GET',
            headers: {
              'auth-token': this._token
            },
            json: true
          };
          const response = await this._httpClient.request(opts);
          if (isDefaultRegion) {
            url = response.url;
          } else {
            url = `https://${response.hostname}.${this._region}.${response.domain}`;
          }
        }
        const isSharedClientApi = [this._url, `https://${this._hostname}.${this._region}.${this._domain}`].includes(url);
        let logMessage = 'Connecting MetaApi websocket client to the MetaApi server ' + `via ${url} ${isSharedClientApi ? 'shared' : 'dedicated'} server.`;
        if (this._firstConnect && !isSharedClientApi) {
          logMessage += ' Please note that it can take up to 3 minutes for your dedicated server to start for the ' + 'first time. During this time it is OK if you see some connection errors.';
          this._firstConnect = false;
        }
        this._logger.info(logMessage);
        return url;
      } catch (error) {
        this._logger.error(error);
        await new _promise2.default(res => setTimeout(res, 1000));
      }
    }
  }

  _throttleRequest(type, accountId, timeInMs) {
    this._lastRequestsTime[type] = this._lastRequestsTime[type] || {};
    let lastTime = this._lastRequestsTime[type][accountId];
    if (!lastTime || lastTime < Date.now() - timeInMs) {
      this._lastRequestsTime[type][accountId] = Date.now();
      return !!lastTime;
    }
    return false;
  }

}
exports.default = MetaApiWebsocketClient;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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