'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _metaApi = require('../metaApi.client');

var _metaApi2 = _interopRequireDefault(_metaApi);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * metaapi.cloud MetaTrader account API client (see https://metaapi.cloud/docs/provisioning/)
 */
class MetatraderAccountClient extends _metaApi2.default {

  /**
   * Extension model
   * @typedef Extension
   * @property {String} id extension id
   * @property {Object} configuration extension configuration
   */

  /**
   * MetaTrader account model
   * @typedef {Object} MetatraderAccountDto
   * @property {String} _id account unique identifier
   * @property {String} name MetaTrader account human-readable name in the MetaApi app
   * @property {String} type account type, can be cloud, cloud-g1, cloud-g2 or self-hosted. Cloud and cloud-g2 are
   * aliases.
   * @property {String} login MetaTrader account number
   * @property {String} server MetaTrader server which hosts the account
   * @property {Version} version MT version
   * @property {String} [provisioningProfileId] id of the account's provisioning profile
   * @property {String} application application name to connect the account to. Currently allowed values are MetaApi and
   * AgiliumTrade
   * @property {Number} magic MetaTrader magic to place trades using
   * @property {String} state account deployment state. One of CREATED, DEPLOYING, DEPLOYED, UNDEPLOYING, UNDEPLOYED,
   * DELETING
   * @property {String} connectionStatus terminal & broker connection status, one of CONNECTED, DISCONNECTED,
   * DISCONNECTED_FROM_BROKER
   * @property {String} accessToken authorization token to be used for accessing single account data.
   * Intended to be used in browser API.
   * @property {Boolean} manualTrades flag indicating if trades should be placed as manual trades. Default is false.
   * Supported on G2 only
   * @property {Number} quoteStreamingIntervalInSeconds Quote streaming interval in seconds. Set to 0 in order to
   * receive quotes on each tick. Default value is 2.5 seconds. Intervals less than 2.5 seconds are supported
   * only for G2
   * @property {Array<string>} [tags] MetaTrader account tags
   * @property {Array<Extension>} [extensions] API extensions
   * @property {Object} [metadata] extra information which can be stored together with your account
   * @property {String} [reliability] used to increase the reliability of the account. Allowed values are regular and
   * high. Default is regular
   * @property {String} [baseCurrency] 3-character ISO currency code of the account base currency. Default value is USD.
   * The setting is to be used for copy trading accounts which use national currencies only, such as some Brazilian
   * brokers. You should not alter this setting unless you understand what you are doing.
   * @property {Array<string>} [copyFactoryRoles] Account roles for CopyFactory2 application. Allowed values are
   * `PROVIDER` and `SUBSCRIBER`
   * @property {Number} [resourceSlots] Number of resource slots to allocate to account. Allocating extra resource slots
   * results in better account performance under load which is useful for some applications. E.g. if you have many
   * accounts copying the same strategy via CooyFactory API, then you can increase resourceSlots to get a lower trade
   * copying latency. Please note that allocating extra resource slots is a paid option. Please note that high
   * reliability accounts use redundant infrastructure, so that each resource slot for a high reliability account
   * is billed as 2 standard resource slots.  Default is 1.
   * @property {number} [copyFactoryResourceSlots] Number of CopyFactory 2 resource slots to allocate to account.
   * Allocating extra resource slots results in lower trade copying latency. Please note that allocating extra resource
   * slots is a paid option. Please also note that CopyFactory 2 uses redundant infrastructure so that
   * each CopyFactory resource slot is billed as 2 standard resource slots. You will be billed for CopyFactory 2
   * resource slots only if you have added your account to CopyFactory 2 by specifying copyFactoryRoles field.
   * Default is 1.
   */

  /**
   * MT version
   * @typedef {4 | 5} Version
   */

  /**
   * Account type
   * @typedef {'cloud' | 'self-hosted'} Type
   */

  /**
   * Account state
   * @typedef {'CREATED' | 'DEPLOYING' | 'DEPLOYED' | 'DEPLOY_FAILED' | 'UNDEPLOYING' | 'UNDEPLOYED' |
   * 'UNDEPLOY_FAILED' | 'DELETING' | 'DELETE_FAILED' | 'REDEPLOY_FAILED'} State
   */

  /**
   * Account connection status
   * @typedef {'CONNECTED' | 'DISCONNECTED' | 'DISCONNECTED_FROM_BROKER'} ConnectionStatus
   */

  /**
   * @typedef {Object} AccountsFilter
   * @property {Number} [offset] search offset (defaults to 0) (must be greater or equal to 0)
   * @property {Number} [limit] search limit (defaults to 1000) 
   * (must be greater or equal to 1 and less or equal to 1000)
   * @property {Array<Version> | Version} [version] MT version
   * @property {Array<Type> | Type} [type] account type
   * @property {Array<State> | State} [state] account state
   * @property {Array<ConnectionStatus> | ConnectionStatus} [connectionStatus] connection status
   * @property {String} [query] searches over _id, name, server and login to match query
   * @property {String} [provisioningProfileId] provisioning profile id
   */

  /**
   * Retrieves MetaTrader accounts owned by user (see https://metaapi.cloud/docs/provisioning/api/account/readAccounts/)
   * Method is accessible only with API access token
   * @param {AccountsFilter} accountsFilter optional filter
   * @return {Promise<Array<MetatraderAccountDto>>} promise resolving with MetaTrader accounts found
   */
  getAccounts(accountsFilter = {}) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('getAccounts');
    }
    const opts = {
      url: `${this._host}/users/current/accounts`,
      method: 'GET',
      qs: accountsFilter,
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Retrieves a MetaTrader account by id (see https://metaapi.cloud/docs/provisioning/api/account/readAccount/). Throws
   * an error if account is not found.
   * @param {String} id MetaTrader account id
   * @return {Promise<MetatraderAccountDto>} promise resolving with MetaTrader account found
   */
  getAccount(id) {
    const opts = {
      url: `${this._host}/users/current/accounts/${id}`,
      method: 'GET',
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Retrieves a MetaTrader account by token (see https://metaapi.cloud/docs/provisioning/api/account/readAccount/).
   * Throws an error if account is not found.
   * Method is accessible only with account access token
   * @return {Promise<MetatraderAccountDto>} promise resolving with MetaTrader account found
   */
  getAccountByToken() {
    if (this._isNotAccountToken()) {
      return this._handleNoAccessError('getAccountByToken');
    }
    const opts = {
      url: `${this._host}/users/current/accounts/accessToken/${this._token}`,
      method: 'GET',
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * New MetaTrader account model
   * @typedef {Object} NewMetatraderAccountDto
   * @property {String} name MetaTrader account human-readable name in the MetaApi app
   * @property {String} type account type, can be cloud, cloud-g1, cloud-g2 or self-hosted. cloud-g2 and cloud are
   * aliases. When you create MT5 cloud account the type is automatically converted to cloud-g1 because MT5 G2 support
   * is still experimental. You can still create MT5 G2 account by setting type to cloud-g2.
   * @property {String} login MetaTrader account number
   * @property {String} password MetaTrader account password. The password can be either investor password for read-only
   * access or master password to enable trading features. Required for cloud account
   * @property {String} server MetaTrader server which hosts the account
   * @property {String} [platform] platform id (mt4 or mt5)
   * @property {String} [provisioningProfileId] id of the account's provisioning profile
   * @property {String} application application name to connect the account to. Currently allowed values are MetaApi and
   * AgiliumTrade
   * @property {Number} magic MetaTrader magic to place trades using
   * @property {Boolean} manualTrades flag indicating if trades should be placed as manual trades. Default is false
   * @property {Number} quoteStreamingIntervalInSeconds Quote streaming interval in seconds. Set to 0 in order to
   * receive quotes on each tick. Default value is 2.5 seconds. Intervals less than 2.5 seconds are supported
   * only for G2
   * @property {Array<string>} [tags] MetaTrader account tags
   * @property {Array<Extension>} extensions API extensions
   * @property {Object} metadata extra information which can be stored together with your account
   * @property {String} reliability used to increase the reliability of the account. Allowed values are regular and high. Default is regular
   * @property {String} baseCurrency 3-character ISO currency code of the account base currency. Default value is USD.
   * The setting is to be used for copy trading accounts which use national currencies only, such as some Brazilian
   * brokers. You should not alter this setting unless you understand what you are doing.
   * @property {Array<string>} copyFactoryRoles Account roles for CopyFactory2 application. Allowed values are
   * `PROVIDER` and `SUBSCRIBER`
   * @property {Number} resourceSlots Number of resource slots to allocate to account. Allocating extra resource slots
   * results in better account performance under load which is useful for some applications. E.g. if you have many
   * accounts copying the same strategy via CooyFactory API, then you can increase resourceSlots to get a lower trade
   * copying latency. Please note that allocating extra resource slots is a paid option. Default is 1
   */

  /**
   * MetaTrader account id model
   * @typedef {Object} MetatraderAccountIdDto
   * @property {String} id MetaTrader account unique identifier
   */

  /**
   * Starts cloud API server for a MetaTrader account using specified provisioning profile (see
   * https://metaapi.cloud/docs/provisioning/api/account/createAccount/). It takes some time to launch the terminal and
   * connect the terminal to the broker, you can use the connectionStatus field to monitor the current status of the
   * terminal.
   * Method is accessible only with API access token
   * @param {NewMetatraderAccountDto} account MetaTrader account to create
   * @return {Promise<MetatraderAccountIdDto>} promise resolving with an id of the MetaTrader account created
   */
  createAccount(account) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('createAccount');
    }
    const opts = {
      url: `${this._host}/users/current/accounts`,
      method: 'POST',
      headers: {
        'auth-token': this._token
      },
      json: true,
      body: account
    };
    return this._httpClient.request(opts);
  }

  /**
   * Starts API server for MetaTrader account. This request will be ignored if the account has already been deployed.
   * (see https://metaapi.cloud/docs/provisioning/api/account/deployAccount/)
   * @param {String} id MetaTrader account id to deploy
   * @return {Promise} promise resolving when MetaTrader account is scheduled for deployment
   */
  deployAccount(id) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('deployAccount');
    }
    const opts = {
      url: `${this._host}/users/current/accounts/${id}/deploy`,
      method: 'POST',
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Stops API server for a MetaTrader account. Terminal data such as downloaded market history data will be preserved.
   * (see https://metaapi.cloud/docs/provisioning/api/account/undeployAccount/)
   * @param {String} id MetaTrader account id to undeploy
   * @return {Promise} promise resolving when MetaTrader account is scheduled for undeployment
   */
  undeployAccount(id) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('undeployAccount');
    }
    const opts = {
      url: `${this._host}/users/current/accounts/${id}/undeploy`,
      method: 'POST',
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Redeploys MetaTrader account. This is equivalent to undeploy immediately followed by deploy.
   * (see https://metaapi.cloud/docs/provisioning/api/account/deployAccount/)
   * @param {String} id MetaTrader account id to redeploy
   * @return {Promise} promise resolving when MetaTrader account is scheduled for redeployment
   */
  redeployAccount(id) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('redeployAccount');
    }
    const opts = {
      url: `${this._host}/users/current/accounts/${id}/redeploy`,
      method: 'POST',
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Stops and deletes an API server for a specified MetaTrader account. The terminal state such as downloaded market
   * data history will be deleted as well when you delete the account. (see
   * https://metaapi.cloud/docs/provisioning/api/account/deleteAccount/).
   * Method is accessible only with API access token
   * @param {String} id MetaTrader account id
   * @return {Promise} promise resolving when MetaTrader account is scheduled for deletion
   */
  deleteAccount(id) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('deleteAccount');
    }
    const opts = {
      url: `${this._host}/users/current/accounts/${id}`,
      method: 'DELETE',
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Updated MetaTrader account data
   * @typedef {Object} MetatraderAccountUpdateDto
   * @property {String} name MetaTrader account human-readable name in the MetaApi app
   * @property {String} password MetaTrader account password. The password can be either investor password for read-only
   * access or master password to enable trading features. Required for cloud account
   * @property {String} server MetaTrader server which hosts the account
   * @property {Boolean} manualTrades flag indicating if trades should be placed as manual trades. Default is false
   * @property {Number} quoteStreamingIntervalInSeconds Quote streaming interval in seconds. Set to 0 in order to
   * receive quotes on each tick. Default value is 2.5 seconds. Intervals less than 2.5 seconds are supported
   * only for G2
   * @property {Array<string>} [tags] MetaTrader account tags
   * @property {Array<Extension>} extensions API extensions
   * @property {Object} metadata extra information which can be stored together with your account
   * @property {Array<string>} copyFactoryRoles Account roles for CopyFactory2 application. Allowed values are
   * `PROVIDER` and `SUBSCRIBER`
   * @property {Number} resourceSlots Number of resource slots to allocate to account. Allocating extra resource slots
   * results in better account performance under load which is useful for some applications. E.g. if you have many
   * accounts copying the same strategy via CooyFactory API, then you can increase resourceSlots to get a lower trade
   * copying latency. Please note that allocating extra resource slots is a paid option. Default is 1
   */

  /**
   * Updates existing metatrader account data (see
   * https://metaapi.cloud/docs/provisioning/api/account/updateAccount/).
   * Method is accessible only with API access token
   * @param {String} id MetaTrader account id
   * @param {MetatraderAccountUpdateDto} account updated MetaTrader account
   * @return {Promise} promise resolving when MetaTrader account is updated
   */
  updateAccount(id, account) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('updateAccount');
    }
    const opts = {
      url: `${this._host}/users/current/accounts/${id}`,
      method: 'PUT',
      headers: {
        'auth-token': this._token
      },
      json: true,
      body: account
    };
    return this._httpClient.request(opts);
  }

  /**
   * Increases MetaTrader account reliability. The account will be temporary stopped to perform this action. (see
   * https://metaapi.cloud/docs/provisioning/api/account/increaseReliability/).
   * Method is accessible only with API access token
   * @param {String} id MetaTrader account id
   * @return {Promise} promise resolving when MetaTrader account reliability is increased
   */
  increaseReliability(id) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('increaseReliability');
    }
    const opts = {
      url: `${this._host}/users/current/accounts/${id}/increase-reliability`,
      method: 'POST',
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

}
exports.default = MetatraderAccountClient;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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