'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _metaApi = require('../metaApi.client');

var _metaApi2 = _interopRequireDefault(_metaApi);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * metaapi.cloud MetaTrader demo account API client
 */
class MetatraderDemoAccountClient extends _metaApi2.default {

  /**
   * New MetaTrader 4 demo account model
   * @typedef {Object} NewMT4DemoAccount
   * @property {String} [accountType] account type
   * @property {String} [address] account holder's address
   * @property {Number} balance account balance
   * @property {String} [city] account holder's city
   * @property {String} [country] account holder's country
   * @property {String} email account holder's email
   * @property {Number} leverage account leverage
   * @property {String} [name] account holder's name
   * @property {String} [phone] account holder's phone
   * @property {String} serverName server name
   * @property {String} [state] account holder's state
   * @property {String} [zip] zip address
   */

  /**
   * New MetaTrader 5 demo account model
   * @typedef {Object} NewMT5DemoAccount
   * @property {String} [address] account holder's address
   * @property {Number} balance account balance
   * @property {String} [city] account holder's city
   * @property {String} [country] account holder's country
   * @property {String} email account holder's email
   * @property {Number} [languageId] language id (default is 1)
   * @property {Number} leverage account leverage
   * @property {String} [name] account holder's name
   * @property {String} [phone] account holder's phone
   * @property {String} serverName server name
   * @property {String} [state] account holder's state
   * @property {String} [zip] zip address
   */

  /**
   * MetaTrader demo account model
   * @typedef {Object} MetatraderDemoAccountDto
   * @property {String} login account login
   * @property {String} password account password
   * @property {String} serverName MetaTrader server name
   * @property {String} investorPassword account investor (read-only) password
   */

  /**
   * Creates new MetaTrader 4 demo account
   * Method is accessible only with API access token
   * @param {String} profileId id of the provisioning profile that will be used as the basis for creating this account
   * @param {NewMT4DemoAccount} account demo account to create
   * @return {Promise<MetatraderDemoAccountDto>} promise resolving with MetaTrader demo account created
   */
  createMT4DemoAccount(profileId, account) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('createMT4DemoAccount');
    }
    const opts = {
      url: `${this._host}/users/current/provisioning-profiles/${profileId}/mt4-demo-accounts`,
      method: 'POST',
      headers: {
        'auth-token': this._token
      },
      json: true,
      body: account
    };
    return this._httpClient.request(opts);
  }

  /**
   * Creates new MetaTrader 5 demo account
   * Method is accessible only with API access token
   * @param {String} profileId id of the provisioning profile that will be used as the basis for creating this account
   * @param {NewMT5DemoAccount} account demo account to create
   * @return {Promise<MetatraderDemoAccountDto>} promise resolving with MetaTrader demo account created
   */
  createMT5DemoAccount(profileId, account) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('createMT5DemoAccount');
    }
    const opts = {
      url: `${this._host}/users/current/provisioning-profiles/${profileId}/mt5-demo-accounts`,
      method: 'POST',
      headers: {
        'auth-token': this._token
      },
      json: true,
      body: account
    };
    return this._httpClient.request(opts);
  }

}
exports.default = MetatraderDemoAccountClient;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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