'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _keys = require('babel-runtime/core-js/object/keys');

var _keys2 = _interopRequireDefault(_keys);

var _stringify = require('babel-runtime/core-js/json/stringify');

var _stringify2 = _interopRequireDefault(_stringify);

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _moment = require('moment');

var _moment2 = _interopRequireDefault(_moment);

var _optionsValidator = require('../optionsValidator');

var _optionsValidator2 = _interopRequireDefault(_optionsValidator);

var _logger = require('../../logger');

var _logger2 = _interopRequireDefault(_logger);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Packet logger options
 * @typedef {Object} PacketLoggerOpts
 * @property {Boolean} [enabled] whether packet logger is enabled
 * @property {Number} [fileNumberLimit] maximum amount of files per account, default value is 12
 * @property {Number} [logFileSizeInHours] amount of logged hours per account file, default value is 4
 * @property {Boolean} [compressSpecifications] whether to compress specifications packets, default value is true
 * @property {Boolean} [compressPrices] whether to compress specifications packets, default value is true
 */

/**
 * A class which records packets into log files
 */
class PacketLogger {

  /**
   * Constructs the class
   * @param {PacketLoggerOpts} opts packet logger options
   */
  constructor(opts) {
    const validator = new _optionsValidator2.default();
    opts = opts || {};
    this._fileNumberLimit = validator.validateNonZero(opts.fileNumberLimit, 12, 'packetLogger.fileNumberLimit');
    this._logFileSizeInHours = validator.validateNonZero(opts.logFileSizeInHours, 4, 'packetLogger.logFileSizeInHours');
    this._compressSpecifications = validator.validateBoolean(opts.compressSpecifications, true, 'packetLogger.compressSpecifications');
    this._compressPrices = validator.validateBoolean(opts.compressPrices, true, 'packetLogger.compressPrices');
    this._previousPrices = {};
    this._lastSNPacket = {};
    this._writeQueue = {};
    this._root = './.metaapi/logs';
    this._logger = _logger2.default.getLogger('PacketLogger');
    _fsExtra2.default.ensureDir(this._root);
  }

  _ensurePreviousPriceObject(accountId) {
    if (!this._previousPrices[accountId]) {
      this._previousPrices[accountId] = {};
    }
  }

  /**
   * Processes packets and pushes them into save queue
   * @param {Object} packet packet to log
   */
  // eslint-disable-next-line complexity
  logPacket(packet) {
    const instanceIndex = packet.instanceIndex || 0;
    if (!this._writeQueue[packet.accountId]) {
      this._writeQueue[packet.accountId] = { isWriting: false, queue: [] };
    }
    if (packet.type === 'status') {
      return;
    }
    if (!this._lastSNPacket[packet.accountId]) {
      this._lastSNPacket[packet.accountId] = {};
    }
    if (['keepalive', 'noop'].includes(packet.type)) {
      this._lastSNPacket[packet.accountId][instanceIndex] = packet;
      return;
    }
    const queue = this._writeQueue[packet.accountId].queue;
    if (!this._previousPrices[packet.accountId]) {
      this._previousPrices[packet.accountId] = {};
    }

    const prevPrice = this._previousPrices[packet.accountId][instanceIndex];

    if (packet.type !== 'prices') {
      if (prevPrice) {
        this._recordPrices(packet.accountId, instanceIndex);
      }
      if (packet.type === 'specifications' && this._compressSpecifications) {
        queue.push((0, _stringify2.default)({ type: packet.type, sequenceNumber: packet.sequenceNumber,
          sequenceTimestamp: packet.sequenceTimestamp, instanceIndex }));
      } else {
        queue.push((0, _stringify2.default)(packet));
      }
    } else {
      if (!this._compressPrices) {
        queue.push((0, _stringify2.default)(packet));
      } else {
        if (prevPrice) {
          const validSequenceNumbers = [prevPrice.last.sequenceNumber, prevPrice.last.sequenceNumber + 1];
          if (this._lastSNPacket[packet.accountId][instanceIndex]) {
            validSequenceNumbers.push(this._lastSNPacket[packet.accountId][instanceIndex].sequenceNumber + 1);
          }
          if (!validSequenceNumbers.includes(packet.sequenceNumber)) {
            this._recordPrices(packet.accountId, instanceIndex);
            this._ensurePreviousPriceObject(packet.accountId);
            this._previousPrices[packet.accountId][instanceIndex] = { first: packet, last: packet };
            queue.push((0, _stringify2.default)(packet));
          } else {
            this._previousPrices[packet.accountId][instanceIndex].last = packet;
          }
        } else {
          this._ensurePreviousPriceObject(packet.accountId);
          this._previousPrices[packet.accountId][instanceIndex] = { first: packet, last: packet };
          queue.push((0, _stringify2.default)(packet));
        }
      }
    }
  }

  /**
   * Returns log messages within date bounds as an array of objects
   * @param {String} accountId account id 
   * @param {Date} dateAfter date to get logs after
   * @param {Date} dateBefore date to get logs before
   * @returns {Array<Object>} log messages
   */
  async readLogs(accountId, dateAfter, dateBefore) {
    const folders = await _fsExtra2.default.readdir(this._root);
    const packets = [];
    for (let folder of folders) {
      const filePath = `${this._root}/${folder}/${accountId}.log`;
      if (await _fsExtra2.default.pathExists(filePath)) {
        const contents = await _fsExtra2.default.readFile(filePath, 'utf8');
        let messages = contents.split('\r\n').filter(message => message.length).map(message => {
          return { date: new Date(message.slice(1, 24)), message: message.slice(26) };
        });
        if (dateAfter) {
          messages = messages.filter(message => message.date > dateAfter);
        }
        if (dateBefore) {
          messages = messages.filter(message => message.date < dateBefore);
        }
        packets.push(...messages);
      }
    }

    return packets;
  }

  /**
   * Returns path for account log file
   * @param {String} accountId account id
   * @returns {String} file path
   */
  async getFilePath(accountId) {
    const fileIndex = Math.floor(new Date().getHours() / this._logFileSizeInHours);
    const folderName = `${(0, _moment2.default)().format('YYYY-MM-DD')}-${fileIndex > 9 ? fileIndex : '0' + fileIndex}`;
    await _fsExtra2.default.ensureDir(`${this._root}/${folderName}`);
    return `${this._root}/${folderName}/${accountId}.log`;
  }

  /**
   * Initializes the packet logger
   */
  start() {
    this._previousPrices = {};
    if (!this._recordInteval) {
      this._recordInteval = setInterval(() => this._appendLogs(), 1000);
      this._deleteOldLogsInterval = setInterval(() => this._deleteOldData(), 10000);
    }
  }

  /**
   * Deinitializes the packet logger
   */
  stop() {
    clearInterval(this._recordInteval);
    clearInterval(this._deleteOldLogsInterval);
  }

  /**
   * Records price packet messages to log files
   * @param {String} accountId account id
   */
  _recordPrices(accountId, instanceNumber) {
    const prevPrice = this._previousPrices[accountId][instanceNumber] || { first: {}, last: {} };
    const queue = this._writeQueue[accountId].queue;
    delete this._previousPrices[accountId][instanceNumber];
    if (!(0, _keys2.default)(this._previousPrices[accountId]).length) {
      delete this._previousPrices[accountId];
    }
    if (prevPrice.first.sequenceNumber !== prevPrice.last.sequenceNumber) {
      queue.push((0, _stringify2.default)(prevPrice.last));
      queue.push(`Recorded price packets ${prevPrice.first.sequenceNumber}` + `-${prevPrice.last.sequenceNumber}, instanceIndex: ${instanceNumber}`);
    }
  }

  /**
   * Writes logs to files
   */
  async _appendLogs() {
    (0, _keys2.default)(this._writeQueue).forEach(async accountId => {
      const queue = this._writeQueue[accountId];
      if (!queue.isWriting && queue.queue.length) {
        queue.isWriting = true;
        try {
          const filePath = await this.getFilePath(accountId);
          const writeString = queue.queue.reduce((a, b) => a + `[${(0, _moment2.default)().format('YYYY-MM-DD HH:mm:ss.SSS')}] ${b}\r\n`, '');
          queue.queue = [];
          await _fsExtra2.default.appendFile(filePath, writeString);
        } catch (err) {
          this._logger.error(`${accountId}: Failed to record packet log`, err);
        }
        queue.isWriting = false;
      }
    });
  }

  /**
   * Deletes folders when the folder limit is exceeded
   */
  async _deleteOldData() {
    const contents = await _fsExtra2.default.readdir(this._root);
    contents.reverse().slice(this._fileNumberLimit).forEach(async folderName => {
      await _fsExtra2.default.remove(`${this._root}/${folderName}`);
    });
  }

}
exports.default = PacketLogger;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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