'use strict';

/**
 * Class which orders the synchronization packets
 */

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _values = require('babel-runtime/core-js/object/values');

var _values2 = _interopRequireDefault(_values);

var _keys = require('babel-runtime/core-js/object/keys');

var _keys2 = _interopRequireDefault(_keys);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class PacketOrderer {

  /**
   * Constructs the class
   * @param {Function} outOfOrderListener function which will receive out of order packet events
   * @param {Number} orderingTimeoutInSeconds packet ordering timeout
   */
  constructor(outOfOrderListener, orderingTimeoutInSeconds) {
    this._outOfOrderListener = outOfOrderListener;
    this._orderingTimeoutInSeconds = orderingTimeoutInSeconds;
    this._isOutOfOrderEmitted = {};
    this._waitListSizeLimit = 100;
    this._sequenceNumberByInstance = {};
    this._lastSessionStartTimestamp = {};
    this._packetsByInstance = {};
  }

  /**
   * Initializes the packet orderer
   */
  start() {
    this._sequenceNumberByInstance = {};
    this._lastSessionStartTimestamp = {};
    this._packetsByInstance = {};
    if (!this._outOfOrderInterval) {
      this._outOfOrderInterval = setInterval(() => this._emitOutOfOrderEvents(), 1000);
    }
  }

  /**
   * Deinitialized the packet orderer
   */
  stop() {
    clearInterval(this._outOfOrderInterval);
  }

  /**
   * Processes the packet and resolves in the order of packet sequence number
   * @param {Object} packet packet to process
   * @return {Array<Object>} ordered packets when the packets are ready to be processed in order
   */
  // eslint-disable-next-line complexity
  restoreOrder(packet) {
    let instanceId = packet.accountId + ':' + (packet.instanceIndex || 0) + ':' + (packet.host || 0);
    if (packet.sequenceNumber === undefined) {
      return [packet];
    }
    if (packet.type === 'synchronizationStarted' && packet.synchronizationId) {
      // synchronization packet sequence just started
      this._isOutOfOrderEmitted[instanceId] = false;
      this._sequenceNumberByInstance[instanceId] = packet.sequenceNumber;
      this._lastSessionStartTimestamp[instanceId] = packet.sequenceTimestamp;
      this._packetsByInstance[instanceId] = (this._packetsByInstance[instanceId] || []).filter(waitPacket => waitPacket.packet.sequenceTimestamp >= packet.sequenceTimestamp);
      return [packet].concat(this._findNextPacketsFromWaitList(instanceId));
    } else if (packet.sequenceTimestamp < this._lastSessionStartTimestamp[instanceId]) {
      // filter out previous packets
      return [];
    } else if (packet.sequenceNumber === this._sequenceNumberByInstance[instanceId]) {
      // let the duplicate s/n packet to pass through
      return [packet];
    } else if (packet.sequenceNumber === this._sequenceNumberByInstance[instanceId] + 1) {
      // in-order packet was received
      this._sequenceNumberByInstance[instanceId]++;
      return [packet].concat(this._findNextPacketsFromWaitList(instanceId));
    } else {
      // out-of-order packet was received, add it to the wait list
      this._packetsByInstance[instanceId] = this._packetsByInstance[instanceId] || [];
      let waitList = this._packetsByInstance[instanceId];
      waitList.push({
        instanceId,
        accountId: packet.accountId,
        instanceIndex: packet.instanceIndex || 0,
        sequenceNumber: packet.sequenceNumber,
        packet: packet,
        receivedAt: new Date()
      });
      waitList.sort((e1, e2) => e1.sequenceNumber - e2.sequenceNumber);
      while (waitList.length > this._waitListSizeLimit) {
        waitList.shift();
      }
      return [];
    }
  }

  /**
   * Resets state for instance id
   * @param {String} instanceId instance id to reset state for
   */
  onStreamClosed(instanceId) {
    delete this._packetsByInstance[instanceId];
    delete this._lastSessionStartTimestamp[instanceId];
    delete this._sequenceNumberByInstance[instanceId];
  }

  /**
   * Resets state for specified accounts on reconnect
   * @param {String[]} reconnectAccountIds reconnected account ids
   */
  onReconnected(reconnectAccountIds) {
    (0, _keys2.default)(this._packetsByInstance).forEach(instanceId => {
      if (reconnectAccountIds.includes(this._getAccountIdFromInstance(instanceId))) {
        delete this._packetsByInstance[instanceId];
      }
    });
    (0, _keys2.default)(this._lastSessionStartTimestamp).forEach(instanceId => {
      if (reconnectAccountIds.includes(this._getAccountIdFromInstance(instanceId))) {
        delete this._lastSessionStartTimestamp[instanceId];
      }
    });
    (0, _keys2.default)(this._sequenceNumberByInstance).forEach(instanceId => {
      if (reconnectAccountIds.includes(this._getAccountIdFromInstance(instanceId))) {
        delete this._sequenceNumberByInstance[instanceId];
      }
    });
  }

  _getAccountIdFromInstance(instanceId) {
    return instanceId.split(':')[0];
  }

  _findNextPacketsFromWaitList(instanceId) {
    let result = [];
    let waitList = this._packetsByInstance[instanceId] || [];
    while (waitList.length && [this._sequenceNumberByInstance[instanceId], this._sequenceNumberByInstance[instanceId] + 1].includes(waitList[0].sequenceNumber)) {
      result.push(waitList[0].packet);
      if (waitList[0].sequenceNumber === this._sequenceNumberByInstance[instanceId] + 1) {
        this._sequenceNumberByInstance[instanceId]++;
      }
      waitList.splice(0, 1);
    }
    if (!waitList.length) {
      delete this._packetsByInstance[instanceId];
    }
    return result;
  }

  _emitOutOfOrderEvents() {
    for (let waitList of (0, _values2.default)(this._packetsByInstance)) {
      if (waitList.length && waitList[0].receivedAt.getTime() + this._orderingTimeoutInSeconds * 1000 < Date.now()) {
        const instanceId = waitList[0].instanceId;
        if (!this._isOutOfOrderEmitted[instanceId]) {
          this._isOutOfOrderEmitted[instanceId] = true;
          // Do not emit onOutOfOrderPacket for packets that come before synchronizationStarted
          if (this._sequenceNumberByInstance[instanceId] !== undefined) {
            this._outOfOrderListener.onOutOfOrderPacket(waitList[0].accountId, waitList[0].instanceIndex, this._sequenceNumberByInstance[instanceId] + 1, waitList[0].sequenceNumber, waitList[0].packet, waitList[0].receivedAt);
          }
        }
      }
    }
  }

}
exports.default = PacketOrderer;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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