'use strict';

var _promise = require('babel-runtime/core-js/promise');

var _promise2 = _interopRequireDefault(_promise);

var _create = require('babel-runtime/core-js/object/create');

var _create2 = _interopRequireDefault(_create);

var _subscriptionManager = require('./subscriptionManager');

var _subscriptionManager2 = _interopRequireDefault(_subscriptionManager);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _timeoutError = require('../timeoutError');

var _timeoutError2 = _interopRequireDefault(_timeoutError);

var _errorHandler = require('../errorHandler');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * @test {SubscriptionManager}
 */
describe('SubscriptionManager', () => {
  let sandbox;
  let clock;
  let manager;
  let client;

  before(() => {
    sandbox = _sinon2.default.createSandbox();
    const mockMath = (0, _create2.default)(global.Math);
    mockMath.random = () => 0.2;
    global.Math = mockMath;
  });

  beforeEach(async () => {
    const socketInstances = [{ socket: { connected: true } }, { socket: { connected: false } }];
    client = {
      connect: () => {},
      connected: socketInstanceIndex => socketInstances[socketInstanceIndex].socket.connected,
      socketInstances: socketInstances,
      socketInstancesByAccounts: { accountId: 0 },
      rpcRequest: () => {}
    };
    clock = _sinon2.default.useFakeTimers({ shouldAdvanceTime: true });
    manager = new _subscriptionManager2.default(client);
  });

  afterEach(async () => {
    sandbox.restore();
    clock.restore();
  });

  /**
   * @test {SubscriptionManager#scheduleSubscribe}
   */
  it('should subscribe to terminal', async () => {
    sandbox.stub(client, 'rpcRequest').resolves();
    setTimeout(() => {
      manager.cancelSubscribe('accountId:0');
    }, 50);
    await manager.scheduleSubscribe('accountId');
    _sinon2.default.assert.calledWith(client.rpcRequest, 'accountId', { type: 'subscribe', instanceIndex: undefined });
  });

  /**
   * @test {SubscriptionManager#scheduleSubscribe}
   */
  it('should retry subscribe if no response received', async () => {
    const response = { type: 'response', accountId: 'accountId', requestId: 'requestId' };
    sandbox.stub(client, 'rpcRequest').onFirstCall().resolves(new _timeoutError2.default('timeout')).onSecondCall().resolves(response).onThirdCall().resolves(response);
    setTimeout(() => {
      manager.cancelSubscribe('accountId:0');
    }, 3600);
    manager.scheduleSubscribe('accountId');
    await clock.tickAsync(10000);
    _sinon2.default.assert.calledTwice(client.rpcRequest);
    _sinon2.default.assert.calledWith(client.rpcRequest, 'accountId', { type: 'subscribe', instanceIndex: undefined });
  });

  /**
   * @test {SubscriptionManager#scheduleSubscribe}
   */
  it('should wait for recommended time if too many requests error received', async () => {
    const response = { type: 'response', accountId: 'accountId', requestId: 'requestId' };
    sandbox.stub(client, 'rpcRequest').onFirstCall().rejects(new _errorHandler.TooManyRequestsError('timeout', {
      periodInMinutes: 60, maxRequestsForPeriod: 10000,
      type: 'LIMIT_REQUEST_RATE_PER_USER',
      recommendedRetryTime: new Date(Date.now() + 5000) })).onSecondCall().resolves(response).onThirdCall().resolves(response);
    manager.scheduleSubscribe('accountId');
    await clock.tickAsync(3600);
    _sinon2.default.assert.callCount(client.rpcRequest, 1);
    await clock.tickAsync(2000);
    manager.cancelSubscribe('accountId:0');
    _sinon2.default.assert.callCount(client.rpcRequest, 2);
  });

  /**
   * @test {SubscriptionManager#onReconnected}
   */
  it('should cancel all subscriptions on reconnect', async () => {
    sandbox.stub(client, 'rpcRequest').resolves();
    client.socketInstancesByAccounts = { accountId: 0, accountId2: 0, accountId3: 1 };
    manager.scheduleSubscribe('accountId');
    manager.scheduleSubscribe('accountId2');
    manager.scheduleSubscribe('accountId3');
    await clock.tickAsync(1000);
    manager.onReconnected(0, []);
    await clock.tickAsync(5000);
    _sinon2.default.assert.callCount(client.rpcRequest, 4);
  });

  /**
   * @test {SubscriptionManager#onReconnected}
   */
  it('should restart subscriptions on reconnect', async () => {
    sandbox.stub(client, 'connect').resolves();
    sandbox.stub(client, 'rpcRequest').resolves();
    client.socketInstancesByAccounts = { accountId: 0, accountId2: 0, accountId3: 0 };
    manager.scheduleSubscribe('accountId');
    manager.scheduleSubscribe('accountId2');
    manager.scheduleSubscribe('accountId3');
    await clock.tickAsync(1000);
    manager.onReconnected(0, ['accountId', 'accountId2']);
    await clock.tickAsync(2000);
    _sinon2.default.assert.callCount(client.rpcRequest, 5);
  });

  /**
   * @test {SubscriptionManager#onReconnected}
   */
  it('should wait until previous subscription ends on reconnect', async () => {
    sandbox.stub(client, 'rpcRequest').callsFake(async () => {
      await new _promise2.default(res => setTimeout(res, 2000));
    });

    sandbox.stub(client, 'connect').resolves();
    client.socketInstancesByAccounts = { accountId: 0 };
    manager.scheduleSubscribe('accountId');
    await clock.tickAsync(1000);
    manager.onReconnected(0, ['accountId']);
    await clock.tickAsync(3000);
    _sinon2.default.assert.callCount(client.rpcRequest, 2);
  });

  /**
   * @test {SubscriptionManager#scheduleSubscribe}
   */
  it('should not send multiple subscribe requests at the same time', async () => {
    sandbox.stub(client, 'rpcRequest').resolves();
    manager.scheduleSubscribe('accountId');
    manager.scheduleSubscribe('accountId');
    await clock.tickAsync(1000);
    manager.cancelSubscribe('accountId:0');
    await clock.tickAsync(2500);
    _sinon2.default.assert.calledWith(client.rpcRequest, 'accountId', { type: 'subscribe', instanceIndex: undefined });
    _sinon2.default.assert.calledOnce(client.rpcRequest);
  });

  /**
   * @test {SubscriptionManager#onTimeout}
   */
  it('should resubscribe on timeout', async () => {
    sandbox.stub(client, 'rpcRequest').resolves();
    client.socketInstances[0].socket.connected = true;
    client.socketInstancesByAccounts.accountId2 = 1;
    setTimeout(() => {
      manager.cancelSubscribe('accountId:0');
      manager.cancelSubscribe('accountId2:0');
    }, 100);
    manager.onTimeout('accountId');
    manager.onTimeout('accountId2');
    await clock.tickAsync(200);
    _sinon2.default.assert.calledWith(client.rpcRequest, 'accountId', { type: 'subscribe', instanceIndex: undefined });
    _sinon2.default.assert.callCount(client.rpcRequest, 1);
  });

  /**
   * @test {SubscriptionManager#onTimeout}
   */
  it('should not retry subscribe to terminal if connection is closed', async () => {
    sandbox.stub(client, 'rpcRequest').resolves();
    client.socketInstances[0].socket.connected = false;
    setTimeout(() => {
      manager.cancelSubscribe('accountId:0');
    }, 100);
    manager.onTimeout('accountId');
    await clock.tickAsync(200);
    _sinon2.default.assert.notCalled(client.rpcRequest);
  });

  /**
   * @test {SubscriptionManager#cancelAccount}
   */
  it('should cancel all subscriptions for an account', async () => {
    sandbox.stub(client, 'rpcRequest').resolves();
    manager.scheduleSubscribe('accountId', 0);
    manager.scheduleSubscribe('accountId', 1);
    await clock.tickAsync(100);
    manager.cancelAccount('accountId');
    await clock.tickAsync(500);
    _sinon2.default.assert.calledTwice(client.rpcRequest);
  });

  /**
   * @test {SubscriptionManager#cancelSubscribe}
   */
  it('should destroy subscribe process on cancel', async () => {
    const subscribe = sandbox.stub().resolves();
    const delaySubscribe = async () => {
      await subscribe();
      await new _promise2.default(res => setTimeout(res, 400));
    };
    client.rpcRequest = delaySubscribe;
    manager.scheduleSubscribe('accountId');
    await clock.tickAsync(50);
    manager.cancelSubscribe('accountId:0');
    await clock.tickAsync(50);
    manager.scheduleSubscribe('accountId');
    await clock.tickAsync(50);
    _sinon2.default.assert.calledTwice(subscribe);
  });

  /**
   * @test {SubscriptionManager#cancelSubscribe}
   */
  it('should check if account is subscribing', async () => {
    manager.scheduleSubscribe('accountId', 1);
    await clock.tickAsync(50);
    _sinon2.default.assert.match(manager.isAccountSubscribing('accountId'), true);
    _sinon2.default.assert.match(manager.isAccountSubscribing('accountId', 0), false);
    _sinon2.default.assert.match(manager.isAccountSubscribing('accountId', 1), true);
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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