'use strict';

var _promise = require('babel-runtime/core-js/promise');

var _promise2 = _interopRequireDefault(_promise);

var _synchronizationThrottler = require('./synchronizationThrottler');

var _synchronizationThrottler2 = _interopRequireDefault(_synchronizationThrottler);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * @test {SynchronizationThrottler}
 */
describe('SynchronizationThrottler', () => {
  let throttler, clock, sandbox, websocketClient;

  before(() => {
    sandbox = _sinon2.default.createSandbox();
  });

  beforeEach(() => {
    clock = _sinon2.default.useFakeTimers({
      now: new Date('2020-10-05T10:00:00.000Z'),
      shouldAdvanceTime: true
    });
    websocketClient = {
      rpcRequest: (accountId, request, timeoutInSeconds) => {},
      subscribedAccountIds: () => new Array(11),
      socketInstances: [{ synchronizationThrottler: { synchronizingAccounts: [] } }]
    };
    websocketClient.rpcRequest = sandbox.stub();
    throttler = new _synchronizationThrottler2.default(websocketClient, 0);
    throttler.start();
  });

  afterEach(() => {
    clock.restore();
  });

  /**
   * @test {SynchronizationThrottler#scheduleSynchronize}
   */
  it('should immediately send request if free slots exist', async () => {
    await throttler.scheduleSynchronize('accountId', { requestId: 'test' });
    _sinon2.default.assert.match(throttler._synchronizationIds, { test: 1601892000000 });
    throttler.removeSynchronizationId('test');
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId', { requestId: 'test' });
    _sinon2.default.assert.match(throttler._synchronizationIds, {});
  });

  /**
   * @test {SynchronizationThrottler#scheduleSynchronize}
   */
  it('should not remove sync if different instance index', async () => {
    await throttler.scheduleSynchronize('accountId', { requestId: 'test', instanceIndex: 0 });
    await throttler.scheduleSynchronize('accountId', { requestId: 'test1', instanceIndex: 1 });
    _sinon2.default.assert.match(throttler._synchronizationIds, { test: 1601892000000, test1: 1601892000000 });
    throttler.removeSynchronizationId('test', 0);
    _sinon2.default.assert.match(throttler._synchronizationIds, { test1: 1601892000000 });
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId', { requestId: 'test', instanceIndex: 0 });
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId', { requestId: 'test1', instanceIndex: 1 });
    _sinon2.default.assert.match(throttler._synchronizationIds, {});
  });

  /**
   * @test {SynchronizationThrottler#scheduleSynchronize}
   */
  it('should wait for other sync requests to finish if slots are full', async () => {
    await throttler.scheduleSynchronize('accountId1', { requestId: 'test1' });
    await throttler.scheduleSynchronize('accountId2', { requestId: 'test2' });
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId1', { requestId: 'test1' });
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId2', { requestId: 'test2' });
    throttler.scheduleSynchronize('accountId3', { requestId: 'test3' });
    await new _promise2.default(res => setTimeout(res, 20));
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 2);
    throttler.removeSynchronizationId('test1');
    await new _promise2.default(res => setTimeout(res, 20));
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 3);
  });

  /**
   * @test {SynchronizationThrottler#scheduleSynchronize}
   */
  it('should increase slot amount with more subscribed accounts', async () => {
    websocketClient.subscribedAccountIds = () => new Array(21);
    await throttler.scheduleSynchronize('accountId1', { requestId: 'test1' });
    await throttler.scheduleSynchronize('accountId2', { requestId: 'test2' });
    await throttler.scheduleSynchronize('accountId3', { requestId: 'test3' });
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId1', { requestId: 'test1' });
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId2', { requestId: 'test2' });
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId3', { requestId: 'test3' });
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 3);
  });

  /**
   * @test {SynchronizationThrottler#scheduleSynchronize}
   */
  it('should set hard limit for concurrent synchronizations across throttlers via options', async () => {
    websocketClient.subscribedAccountIds = () => new Array(21);
    throttler = new _synchronizationThrottler2.default(websocketClient, 0, { maxConcurrentSynchronizations: 3 });
    websocketClient.socketInstances = [{ synchronizationThrottler: throttler }, { synchronizationThrottler: { synchronizingAccounts: ['accountId4'] } }];
    await throttler.scheduleSynchronize('accountId1', { requestId: 'test1' });
    await throttler.scheduleSynchronize('accountId2', { requestId: 'test2' });
    throttler.scheduleSynchronize('accountId3', { requestId: 'test3' });
    throttler.scheduleSynchronize('accountId4', { requestId: 'test4' });
    await new _promise2.default(res => setTimeout(res, 20));
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 2);
    throttler.removeSynchronizationId('test1');
    await new _promise2.default(res => setTimeout(res, 20));
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 3);
  });

  /**
   * @test {SynchronizationThrottler#scheduleSynchronize}
   */
  it('should not take extra slots if sync ids belong to the same account', async () => {
    throttler.scheduleSynchronize('accountId', { requestId: 'test', instanceIndex: 0 });
    throttler.scheduleSynchronize('accountId', { requestId: 'test1', instanceIndex: 1 });
    throttler.scheduleSynchronize('accountId2', { requestId: 'test2' });
    throttler.scheduleSynchronize('accountId3', { requestId: 'test3' });
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 3);
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId', { requestId: 'test', instanceIndex: 0 });
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId', { requestId: 'test1', instanceIndex: 1 });
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId2', { requestId: 'test2' });
    _sinon2.default.assert.match(throttler._synchronizationIds, {});
  });

  /**
   * @test {SynchronizationThrottler#_removeOldSyncIdsJob}
   */
  it('should clear expired synchronization slots if no packets for 10 seconds', async () => {
    await throttler.scheduleSynchronize('accountId1', { requestId: 'test1' });
    await throttler.scheduleSynchronize('accountId2', { requestId: 'test2' });
    throttler.scheduleSynchronize('accountId3', { requestId: 'test3' });
    await new _promise2.default(res => setTimeout(res, 20));
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 2);
    await clock.tickAsync(11000);
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 3);
  });

  /**
   * @test {SynchronizationThrottler#updateSynchronizationId}
   */
  it('should renew sync on update', async () => {
    await throttler.scheduleSynchronize('accountId1', { requestId: 'test1' });
    await throttler.scheduleSynchronize('accountId2', { requestId: 'test2' });
    throttler.scheduleSynchronize('accountId3', { requestId: 'test3' });
    await new _promise2.default(res => setTimeout(res, 20));
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 2);
    await clock.tickAsync(11000);
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 3);
    await clock.tickAsync(11000);
    throttler.updateSynchronizationId('test1');
    throttler.scheduleSynchronize('accountId4', { requestId: 'test4' });
    throttler.scheduleSynchronize('accountId5', { requestId: 'test5' });
    await new _promise2.default(res => setTimeout(res, 20));
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 4);
  });

  /**
   * @test {SynchronizationThrottler#scheduleSynchronize}
   */
  it('should replace previous syncs', async () => {
    throttler.scheduleSynchronize('accountId1', { requestId: 'test1' });
    throttler.scheduleSynchronize('accountId1', { requestId: 'test2' });
    throttler.scheduleSynchronize('accountId1', { requestId: 'test3' });
    throttler.scheduleSynchronize('accountId2', { requestId: 'test4' });
    throttler.scheduleSynchronize('accountId3', { requestId: 'test5' });
    throttler.scheduleSynchronize('accountId1', { requestId: 'test3', instanceIndex: 0 });
    await new _promise2.default(res => setTimeout(res, 20));
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 4);
  });

  /**
   * @test {SynchronizationThrottler#onDisconnect}
   */
  it('should clear existing sync ids on disconnect', async () => {
    await throttler.scheduleSynchronize('accountId1', { requestId: 'test1' });
    await throttler.scheduleSynchronize('accountId2', { requestId: 'test2' });
    await new _promise2.default(res => setTimeout(res, 20));
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 2);
    throttler.onDisconnect();
    throttler.scheduleSynchronize('accountId3', { requestId: 'test3' });
    await new _promise2.default(res => setTimeout(res, 20));
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 3);
  });

  /**
   * @test {SynchronizationThrottler#_removeFromQueue}
   */
  it('should remove synchronizations from queue', async () => {
    await throttler.scheduleSynchronize('accountId1', { requestId: 'test1' });
    await throttler.scheduleSynchronize('accountId2', { requestId: 'test2' });
    throttler.scheduleSynchronize('accountId3', { requestId: 'test3' });
    throttler.scheduleSynchronize('accountId3', { requestId: 'test4', instanceIndex: 0 });
    throttler.scheduleSynchronize('accountId4', { requestId: 'test5' });
    throttler.scheduleSynchronize('accountId3', { requestId: 'test6' });
    throttler.scheduleSynchronize('accountId4', { requestId: 'test7' });
    throttler.scheduleSynchronize('accountId3', { requestId: 'test8' });
    throttler.scheduleSynchronize('accountId5', { requestId: 'test9' });
    throttler.scheduleSynchronize('accountId3', { requestId: 'test10', instanceIndex: 0 });
    await clock.tickAsync(53000);
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 6);
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId1', { requestId: 'test1' });
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId2', { requestId: 'test2' });
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId3', { requestId: 'test8' });
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId3', { requestId: 'test10', instanceIndex: 0 });
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId4', { requestId: 'test7' });
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId5', { requestId: 'test9' });
  });

  /**
   * @test {SynchronizationThrottler#_removeOldSyncIdsJob}
   */
  it('should remove expired synchronizations from queue', async () => {
    await throttler.scheduleSynchronize('accountId1', { requestId: 'test1' });
    await throttler.scheduleSynchronize('accountId2', { requestId: 'test2' });
    throttler.scheduleSynchronize('accountId3', { requestId: 'test3' });
    throttler.scheduleSynchronize('accountId4', { requestId: 'test4' });
    for (let i = 0; i < 20; i++) {
      await clock.tickAsync(8000);
      throttler.updateSynchronizationId('test1');
      throttler.updateSynchronizationId('test2');
    }
    throttler.scheduleSynchronize('accountId5', { requestId: 'test5' });
    for (let i = 0; i < 20; i++) {
      await clock.tickAsync(8000);
      throttler.updateSynchronizationId('test1');
      throttler.updateSynchronizationId('test2');
    }
    await clock.tickAsync(33000);
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 3);
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId1', { requestId: 'test1' });
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId2', { requestId: 'test2' });
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId5', { requestId: 'test5' });
  });

  /**
   * @test {SynchronizationThrottler#scheduleSynchronize}
   */
  it('should not get queue stuck due to app synchronizations limit', async () => {
    throttler._client.socketInstances = [{ synchronizationThrottler: { synchronizingAccounts: ['accountId21', 'accountId22', 'accountId23', 'accountId24', 'accountId25', 'accountId26', 'accountId27', 'accountId28', 'accountId29', 'accountId210', 'accountId211', 'accountId212', 'accountId213', 'accountId214', 'accountId215'] } }, { synchronizationThrottler: throttler }];
    throttler.scheduleSynchronize('accountId1', { requestId: 'test1' });
    throttler.scheduleSynchronize('accountId2', { requestId: 'test2' });
    throttler.scheduleSynchronize('accountId3', { requestId: 'test3' });
    await clock.tickAsync(5000);
    _sinon2.default.assert.notCalled(websocketClient.rpcRequest);
    throttler._client.socketInstances[0].synchronizationThrottler.synchronizingAccounts = throttler._client.socketInstances[0].synchronizationThrottler.synchronizingAccounts.slice(1);
    await clock.tickAsync(5000);
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 1);
    throttler._client.socketInstances[0].synchronizationThrottler.synchronizingAccounts = throttler._client.socketInstances[0].synchronizationThrottler.synchronizingAccounts.slice(1);
    await clock.tickAsync(5000);
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 2);
  });

  /**
   * @test {SynchronizationThrottler#removeSynchronizationId}
   */
  it('should not skip queue items when synchronization id is removed', async () => {
    throttler.scheduleSynchronize('accountId1', { requestId: 'test1' });
    throttler.scheduleSynchronize('accountId2', { requestId: 'test2' });
    throttler.scheduleSynchronize('accountId3', { requestId: 'test3' });
    throttler.scheduleSynchronize('accountId4', { requestId: 'test4' });
    throttler.scheduleSynchronize('accountId5', { requestId: 'test5' });
    await clock.tickAsync(2000);
    throttler.removeSynchronizationId('test3');
    await clock.tickAsync(2000);
    throttler.removeSynchronizationId('test1');
    throttler.removeSynchronizationId('test2');
    await clock.tickAsync(2000);
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 4);
  });

  /**
   * @test {SynchronizationThrottler#removeIdByParameters}
   */
  it('should remove id by parameters', async () => {
    await throttler.scheduleSynchronize('accountId1', { requestId: 'test1' });
    await throttler.scheduleSynchronize('accountId2', { requestId: 'test2', instanceIndex: 0, host: 'ps-mpa-0' });
    throttler.scheduleSynchronize('accountId3', { requestId: 'test3' });
    throttler.scheduleSynchronize('accountId2', { requestId: 'test4', instanceIndex: 1, host: 'ps-mpa-1' });
    throttler.scheduleSynchronize('accountId2', { requestId: 'test5', instanceIndex: 0, host: 'ps-mpa-2' });
    throttler.scheduleSynchronize('accountId4', { requestId: 'test6' });
    await new _promise2.default(res => setTimeout(res, 50));
    throttler.scheduleSynchronize('accountId2', { requestId: 'test7', instanceIndex: 0, host: 'ps-mpa-3' });
    await new _promise2.default(res => setTimeout(res, 50));
    throttler.removeIdByParameters('accountId2', 0, 'ps-mpa-0');
    throttler.removeIdByParameters('accountId2', 0, 'ps-mpa-3');
    throttler.removeIdByParameters('accountId2', 1, 'ps-mpa-1');
    throttler.removeSynchronizationId('test1');
    await new _promise2.default(res => setTimeout(res, 50));
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId3', { requestId: 'test3' });
    _sinon2.default.assert.calledWith(websocketClient.rpcRequest, 'accountId2', { requestId: 'test5', instanceIndex: 0,
      host: 'ps-mpa-2' });
    _sinon2.default.assert.callCount(websocketClient.rpcRequest, 4);
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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