'use strict';

var _keys = require('babel-runtime/core-js/object/keys');

var _keys2 = _interopRequireDefault(_keys);

var _values = require('babel-runtime/core-js/object/values');

var _values2 = _interopRequireDefault(_values);

var _create = require('babel-runtime/core-js/object/create');

var _create2 = _interopRequireDefault(_create);

var _promise = require('babel-runtime/core-js/promise');

var _promise2 = _interopRequireDefault(_promise);

var _should = require('should');

var _should2 = _interopRequireDefault(_should);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _socket = require('socket.io');

var _socket2 = _interopRequireDefault(_socket);

var _metaApi = require('../metaApi/metaApi');

var _metaApi2 = _interopRequireDefault(_metaApi);

var _synchronizationListener = require('../clients/metaApi/synchronizationListener');

var _synchronizationListener2 = _interopRequireDefault(_synchronizationListener);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const accountInformation = {
  broker: 'True ECN Trading Ltd',
  currency: 'USD',
  server: 'ICMarketsSC-Demo',
  balance: 7319.9,
  equity: 7306.649913200001,
  margin: 184.1,
  freeMargin: 7120.22,
  leverage: 100,
  marginLevel: 3967.58283542
};
const errors = [{
  'id': 1,
  'error': 'TooManyRequestsError',
  'message': 'One user can connect to one server no more than 300 accounts. Current number of connected ' + 'accounts 300. For more information see https://metaapi.cloud/docs/client/rateLimiting/',
  'metadata': {
    'maxAccountsPerUserPerServer': 300,
    'accountsCount': 300,
    'recommendedRetryTime': new Date(Date.now() + 20000),
    'type': 'LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_USER_PER_SERVER'
  }
}, {
  'id': 1,
  'error': 'TooManyRequestsError',
  'message': 'You have used all your account subscriptions quota. You have 50 account subscriptions available ' + 'and have used 50 subscriptions. Please deploy more accounts to get more subscriptions. For more ' + 'information see https://metaapi.cloud/docs/client/rateLimiting/',
  'metadata': {
    'maxAccountsPerUser': 50,
    'accountsCount': 50,
    'recommendedRetryTime': new Date(Date.now() + 20000),
    'type': 'LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_USER'
  }
}, {
  'id': 1,
  'error': 'TooManyRequestsError',
  'message': 'You can not subscribe to more accounts on this connection because server is out of capacity. ' + 'Please establish a new connection with a different client-id header value to switch to a ' + 'different server. For more information see https://metaapi.cloud/docs/client/rateLimiting/',
  'metadata': {
    'changeClientIdHeader': true,
    'recommendedRetryTime': new Date(Date.now() + 20000),
    'type': 'LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_SERVER'
  }
}];

let server;

class FakeServer {
  constructor() {
    this.io;
    this.socket;
    this.statusTasks = {};
  }

  async authenticate(socket, data, host = 'ps-mpa-0') {
    socket.emit('synchronization', { type: 'authenticated', accountId: data.accountId,
      instanceIndex: 0, replicas: 1, host });
  }

  deleteStatusTask(accountId) {
    if (this.statusTasks[accountId]) {
      clearInterval(this.statusTasks[accountId]);
      delete this.statusTasks[accountId];
    }
  }

  async emitStatus(socket, accountId, host = 'ps-mpa-0') {
    const packet = { connected: true, authenticated: true, instanceIndex: 0, type: 'status',
      healthStatus: { rpcApiHealthy: true }, replicas: 1, host,
      connectionId: accountId, accountId: accountId };
    socket.emit('synchronization', packet);
  }

  async respondAccountInformation(socket, data) {
    await socket.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId,
      accountInformation });
  }

  async syncAccount(socket, data, host = 'ps-mpa-0') {
    socket.emit('synchronization', { type: 'synchronizationStarted', accountId: data.accountId, instanceIndex: 0,
      synchronizationId: data.requestId, host });
    await new _promise2.default(res => setTimeout(res, 50));
    socket.emit('synchronization', { type: 'accountInformation', accountId: data.accountId, accountInformation,
      instanceIndex: 0, host });
    socket.emit('synchronization', { type: 'specifications', accountId: data.accountId, specifications: [], instanceIndex: 0, host });
    socket.emit('synchronization', { type: 'positions', accountId: data.accountId, synchronizationId: data.requestId,
      positions: [], instanceIndex: 0, host });
    socket.emit('synchronization', { type: 'orders', accountId: data.accountId, synchronizationId: data.requestId,
      orders: [], instanceIndex: 0, host });
    await new _promise2.default(res => setTimeout(res, 50));
    socket.emit('synchronization', { type: 'orderSynchronizationFinished', accountId: data.accountId,
      synchronizationId: data.requestId, instanceIndex: 0, host });
    await new _promise2.default(res => setTimeout(res, 50));
    socket.emit('synchronization', { type: 'dealSynchronizationFinished', accountId: data.accountId,
      synchronizationId: data.requestId, instanceIndex: 0, host });
  }

  async respond(socket, data) {
    await socket.emit('response', { type: 'response', accountId: data.accountId, requestId: data.requestId });
  }

  async emitError(socket, data, errorIndex, retryAfterSeconds) {
    const error = errors[errorIndex];
    error.metadata.recommendedRetryTime = new Date(Date.now() + retryAfterSeconds * 1000);
    error.requestId = data.requestId;
    await socket.emit('processingError', error);
  }

  enableSync(socket) {
    socket.removeAllListeners('request');
    socket.on('request', async data => {
      if (data.type === 'subscribe') {
        await new _promise2.default(res => setTimeout(res, 200));
        await this.respond(socket, data);
        this.statusTasks[data.accountId] = setInterval(() => this.emitStatus(socket, data.accountId), 100);
        await new _promise2.default(res => setTimeout(res, 50));
        await this.authenticate(socket, data);
      } else if (data.type === 'synchronize') {
        await this.respond(socket, data);
        await new _promise2.default(res => setTimeout(res, 50));
        await this.syncAccount(socket, data);
      } else if (data.type === 'waitSynchronized') {
        await this.respond(socket, data);
      } else if (data.type === 'getAccountInformation') {
        await this.respondAccountInformation(socket, data);
      } else if (data.type === 'unsubscribe') {
        this.deleteStatusTask(data.accountId);
        await this.respond(socket, data);
      }
    });
  }

  disableSync() {
    server.removeAllListeners('request');
    server.on('request', async data => {
      await this.respond(server, data);
    });
  }

  async start(port = 6785) {
    this.io = new _socket2.default(port, { path: '/ws', pingTimeout: 1000000 });
    this.io.on('connect', socket => {
      server = socket;
      socket.emit('response', { type: 'response' });
      this.enableSync(socket);
    });
  }

}

const sequentialProcessing = [true];
sequentialProcessing.forEach(param => {
  describe('Synchronization stability test', () => {

    let fakeServer;
    let connection;
    let clock;
    let sandbox;
    let api;

    before(() => {
      sandbox = _sinon2.default.createSandbox();
      const mockMath = (0, _create2.default)(global.Math);
      mockMath.random = () => 0.2;
      global.Math = mockMath;
    });

    beforeEach(async () => {
      clock = _sinon2.default.useFakeTimers({ shouldAdvanceTime: true });
      api = new _metaApi2.default('token', { application: 'application', domain: 'project-stock.agiliumlabs.cloud',
        useSharedClientApi: true, requestTimeout: 3, retryOpts: {
          retries: 3, minDelayInSeconds: 0.1, maxDelayInSeconds: 0.5, subscribeCooldownInSeconds: 6 },
        eventProcessing: { sequentialProcessing: param } });
      api.metatraderAccountApi._metatraderAccountClient.getAccount = accountId => ({
        _id: accountId,
        login: '50194988',
        name: 'mt5a',
        server: 'ICMarketsSC-Demo',
        provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
        magic: 123456,
        application: 'MetaApi',
        connectionStatus: 'DISCONNECTED',
        state: 'DEPLOYED',
        type: 'cloud',
        accessToken: '2RUnoH1ldGbnEneCoqRTgI4QO1XOmVzbH5EVoQsA'
      });
      api._metaApiWebsocketClient.url = 'http://localhost:6785';
      fakeServer = new FakeServer();
      await fakeServer.start();
    });

    afterEach(async () => {
      (0, _values2.default)(fakeServer.statusTasks).forEach(task => clearInterval(task));
      connection._websocketClient._subscriptionManager.cancelAccount('accountId');
      connection._websocketClient.close();
      let resolve;
      let promise = new _promise2.default(res => resolve = res);
      fakeServer.io.close(() => {
        resolve();
      });
      await promise;
      sandbox.restore();
      clock.restore();
      await new _promise2.default(res => setTimeout(res, 50));
    });

    it('should synchronize account', async () => {
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      clock.tickAsync(5000);
      await connection.waitSynchronized({ timeoutInSeconds: 10 });
      const response = connection.terminalState.accountInformation;
      _sinon2.default.assert.match(response, accountInformation);
      (connection.synchronized && connection.terminalState.connected && connection.terminalState.connectedToBroker).should.equal(true);
    }).timeout(10000);

    it('should reconnect on server socket crash', async () => {
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      clock.tickAsync(5000);
      await connection.waitSynchronized({ timeoutInSeconds: 10 });
      server.disconnect();
      await new _promise2.default(res => setTimeout(res, 50));
      await clock.tickAsync(10000);
      const response = connection.terminalState.accountInformation;
      _sinon2.default.assert.match(response, accountInformation);
      (connection.synchronized && connection.terminalState.connected && connection.terminalState.connectedToBroker).should.equal(true);
    }).timeout(10000);

    it('should set state to disconnected on timeout', async () => {
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      clock.tickAsync(5000);
      await connection.waitSynchronized({ timeoutInSeconds: 10 });
      fakeServer.deleteStatusTask('accountId');
      fakeServer.io.on('connect', socket => {
        socket.disconnect();
      });
      server.disconnect();
      await clock.tickAsync(61000);
      await new _promise2.default(res => setTimeout(res, 50));
      connection.synchronized.should.equal(false);
      connection.terminalState.connected.should.equal(false);
      connection.terminalState.connectedToBroker.should.equal(false);
    }).timeout(10000);

    it('should resubscribe on timeout', async () => {
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      clock.tickAsync(5000);
      await connection.waitSynchronized({ timeoutInSeconds: 10 });
      fakeServer.deleteStatusTask('accountId');
      await clock.tickAsync(61000);
      await new _promise2.default(res => setTimeout(res, 50));
      const response = connection.terminalState.accountInformation;
      _sinon2.default.assert.match(response, accountInformation);
      (connection.synchronized && connection.terminalState.connected && connection.terminalState.connectedToBroker).should.equal(true);
    }).timeout(10000);

    it('should synchronize if subscribe response arrives after synchronization', async () => {
      fakeServer.io.removeAllListeners('connect');
      fakeServer.io.on('connect', socket => {
        server = socket;
        socket.emit('response', { type: 'response' });
        socket.removeAllListeners('request');
        socket.on('request', async data => {
          if (data.type === 'subscribe') {
            await new _promise2.default(res => setTimeout(res, 200));
            fakeServer.statusTasks.accountId = setInterval(() => fakeServer.emitStatus(socket, data.accountId), 100);
            await fakeServer.authenticate(socket, data);
            await new _promise2.default(res => setTimeout(res, 400));
            await fakeServer.respond(socket, data);
          } else if (data.type === 'synchronize') {
            await fakeServer.respond(socket, data);
            await fakeServer.syncAccount(socket, data);
          } else if (data.type === 'waitSynchronized') {
            await fakeServer.respond(socket, data);
          } else if (data.type === 'getAccountInformation') {
            await fakeServer.respondAccountInformation(socket, data);
          }
        });
      });
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      clock.tickAsync(5000);
      await connection.waitSynchronized({ timeoutInSeconds: 10 });
      const response = connection.terminalState.accountInformation;
      _sinon2.default.assert.match(response, accountInformation);
      (connection.synchronized && connection.terminalState.connected && connection.terminalState.connectedToBroker).should.equal(true);
    }).timeout(10000);

    it('should wait until account is redeployed after disconnect', async () => {
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      clock.tickAsync(5000);
      await connection.waitSynchronized({ timeoutInSeconds: 10 });
      fakeServer.deleteStatusTask('accountId');
      fakeServer.disableSync();
      await server.emit('synchronization', { type: 'disconnected', accountId: 'accountId',
        host: 'ps-mpa-0', instanceIndex: 0 });
      await clock.tickAsync(20000);
      connection.synchronized.should.equal(false);
      connection.terminalState.connected.should.equal(false);
      connection.terminalState.connectedToBroker.should.equal(false);
      await clock.tickAsync(200000);
      fakeServer.enableSync(server);
      await clock.tickAsync(20000);
      connection.synchronized.should.equal(false);
      connection.terminalState.connected.should.equal(false);
      connection.terminalState.connectedToBroker.should.equal(false);
      await clock.tickAsync(200000);
      (connection.synchronized && connection.terminalState.connected && connection.terminalState.connectedToBroker).should.equal(true);
    }).timeout(10000);

    it('should resubscribe immediately after disconnect on status packet', async () => {
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      clock.tickAsync(5000);
      await connection.waitSynchronized({ timeoutInSeconds: 10 });
      fakeServer.deleteStatusTask('accountId');
      fakeServer.disableSync();
      await server.emit('synchronization', { type: 'disconnected', accountId: 'accountId',
        host: 'ps-mpa-0', instanceIndex: 0 });
      await clock.tickAsync(20000);
      connection.synchronized.should.equal(false);
      connection.terminalState.connected.should.equal(false);
      connection.terminalState.connectedToBroker.should.equal(false);
      await clock.tickAsync(200000);
      fakeServer.enableSync(server);
      await fakeServer.emitStatus(server, 'accountId');
      await clock.tickAsync(20000);
      (connection.synchronized && connection.terminalState.connected && connection.terminalState.connectedToBroker).should.equal(true);
    }).timeout(10000);

    it('should reconnect after server restarts', async () => {
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      await new _promise2.default(res => setTimeout(res, 50));
      clock.tickAsync(5000);
      await connection.waitSynchronized({ timeoutInSeconds: 10 });
      for (let i = 0; i < 5; i++) {
        fakeServer.deleteStatusTask('accountId');
        fakeServer.io.close();
        await clock.tickAsync(200000);
        await new _promise2.default(res => setTimeout(res, 50));
        clock.tickAsync(2000);
        await fakeServer.start();
        await new _promise2.default(res => setTimeout(res, 50));
        await clock.tickAsync(1000);
      }
      await clock.tickAsync(20000);
      await new _promise2.default(res => setTimeout(res, 100));
      await clock.tickAsync(20000);
      const response = connection.terminalState.accountInformation;
      _sinon2.default.assert.match(response, accountInformation);
      connection.synchronized.should.equal(true);
      connection.terminalState.connected.should.equal(true);
      connection.terminalState.connectedToBroker.should.equal(true);
    }).timeout(20000);

    it('should synchronize if connecting while server is rebooting', async () => {
      fakeServer.io.close();
      await fakeServer.start(9000);
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      setTimeout(() => {
        fakeServer.io.close();
        fakeServer.start();
      }, 3000);
      await new _promise2.default(res => setTimeout(res, 50));
      clock.tickAsync(7000);
      await connection.waitSynchronized({ timeoutInSeconds: 10 });
      const response = connection.terminalState.accountInformation;
      _sinon2.default.assert.match(response, accountInformation);
      (connection.synchronized && connection.terminalState.connected && connection.terminalState.connectedToBroker).should.equal(true);
    }).timeout(10000);

    it('should resubscribe other accounts after one of connections is closed', async () => {
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      clock.tickAsync(5000);
      await connection.waitSynchronized({ timeoutInSeconds: 10 });
      const account2 = await api.metatraderAccountApi.getAccount('accountId2');
      const connection2 = account2.getStreamingConnection();
      await connection2.connect();
      clock.tickAsync(5000);
      await connection2.waitSynchronized({ timeoutInSeconds: 10 });
      const account3 = await api.metatraderAccountApi.getAccount('accountId3');
      const connection3 = account3.getStreamingConnection();
      await connection3.connect();
      clock.tickAsync(1000);
      await connection3.waitSynchronized({ timeoutInSeconds: 10 });
      await connection.close();
      fakeServer.deleteStatusTask('accountId2');
      fakeServer.deleteStatusTask('accountId3');
      fakeServer.disableSync();
      server.disconnect();
      await clock.tickAsync(2000);
      await new _promise2.default(res => setTimeout(res, 50));
      fakeServer.enableSync(server);
      server.disconnect();
      await clock.tickAsync(8000);
      await new _promise2.default(res => setTimeout(res, 50));
      connection.synchronized.should.equal(false);
      connection2.synchronized.should.equal(true);
      connection2.terminalState.connected.should.equal(true);
      connection2.terminalState.connectedToBroker.should.equal(true);
      connection3.synchronized.should.equal(true);
      connection3.terminalState.connected.should.equal(true);
      connection3.terminalState.connectedToBroker.should.equal(true);
    });

    it('should limit subscriptions during per user 429 error', async () => {
      const subscribedAccounts = {};
      fakeServer.io.removeAllListeners('connect');
      fakeServer.io.on('connect', socket => {
        server = socket;
        socket.emit('response', { type: 'response' });
        socket.removeAllListeners('request');
        socket.on('request', async data => {
          if (data.type === 'subscribe') {
            if ((0, _keys2.default)(subscribedAccounts).length < 2) {
              subscribedAccounts[data.accountId] = true;
              await new _promise2.default(res => setTimeout(res, 200));
              await fakeServer.respond(socket, data);
              fakeServer.statusTasks.accountId = setInterval(() => fakeServer.emitStatus(socket, data.accountId), 100);
              await new _promise2.default(res => setTimeout(res, 50));
              await fakeServer.authenticate(socket, data);
            } else {
              await fakeServer.emitError(socket, data, 1, 2);
            }
          } else if (data.type === 'synchronize') {
            await fakeServer.respond(socket, data);
            await fakeServer.syncAccount(socket, data);
          } else if (data.type === 'waitSynchronized') {
            await fakeServer.respond(socket, data);
          } else if (data.type === 'getAccountInformation') {
            await fakeServer.respondAccountInformation(socket, data);
          } else if (data.type === 'unsubscribe') {
            delete subscribedAccounts[data.accountId];
            await fakeServer.respondAccountInformation(socket, data);
          }
        });
      });
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      await new _promise2.default(res => setTimeout(res, 50));
      clock.tickAsync(5000);
      await connection.waitSynchronized({ timeoutInSeconds: 5 });
      const account2 = await api.metatraderAccountApi.getAccount('accountId2');
      const connection2 = account2.getStreamingConnection();
      await connection2.connect();
      await new _promise2.default(res => setTimeout(res, 50));
      clock.tickAsync(5000);
      await connection2.waitSynchronized({ timeoutInSeconds: 5 });
      const account3 = await api.metatraderAccountApi.getAccount('accountId3');
      const connection3 = account3.getStreamingConnection();
      await connection3.connect();
      try {
        await new _promise2.default(res => setTimeout(res, 50));
        clock.tickAsync(5000);
        await connection3.waitSynchronized({ timeoutInSeconds: 5 });
        throw new Error('TimeoutError expected');
      } catch (err) {
        err.name.should.equal('TimeoutError');
      }
      await connection2.close();
      await clock.tickAsync(2000);
      _sinon2.default.assert.match(connection3.synchronized, true);
    }).timeout(10000);

    it('should wait for retry time after per user 429 error', async () => {
      let requestTimestamp = 0;
      const subscribedAccounts = {};

      fakeServer.io.removeAllListeners('connect');
      fakeServer.io.on('connect', socket => {
        server = socket;
        socket.emit('response', { type: 'response' });
        socket.removeAllListeners('request');
        //eslint-disable-next-line complexity
        socket.on('request', async data => {
          if (data.type === 'subscribe') {
            if ((0, _keys2.default)(subscribedAccounts).length < 2 || requestTimestamp !== 0 && Date.now() - 2 * 1000 > requestTimestamp) {
              subscribedAccounts[data.accountId] = true;
              await new _promise2.default(res => setTimeout(res, 200));
              await fakeServer.respond(socket, data);
              fakeServer.statusTasks.accountId = setInterval(() => fakeServer.emitStatus(socket, data.accountId), 100);
              await new _promise2.default(res => setTimeout(res, 50));
              await fakeServer.authenticate(socket, data);
            } else {
              requestTimestamp = Date.now();
              await fakeServer.emitError(socket, data, 1, 3);
            }
          } else if (data.type === 'synchronize') {
            await fakeServer.respond(socket, data);
            await fakeServer.syncAccount(socket, data);
          } else if (data.type === 'waitSynchronized') {
            await fakeServer.respond(socket, data);
          } else if (data.type === 'getAccountInformation') {
            await fakeServer.respondAccountInformation(socket, data);
          } else if (data.type === 'unsubscribe') {
            delete subscribedAccounts[data.accountId];
            await fakeServer.respond(socket, data);
          }
        });
      });
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      await new _promise2.default(res => setTimeout(res, 50));
      clock.tickAsync(1500);
      await connection.waitSynchronized({ timeoutInSeconds: 3 });
      const account2 = await api.metatraderAccountApi.getAccount('accountId2');
      const connection2 = account2.getStreamingConnection();
      await connection2.connect();
      await new _promise2.default(res => setTimeout(res, 50));
      clock.tickAsync(1500);
      await connection2.waitSynchronized({ timeoutInSeconds: 3 });
      const account3 = await api.metatraderAccountApi.getAccount('accountId3');
      const connection3 = account3.getStreamingConnection();
      await connection3.connect();
      try {
        await new _promise2.default(res => setTimeout(res, 50));
        clock.tickAsync(3000);
        await connection3.waitSynchronized({ timeoutInSeconds: 3 });
        throw new Error('TimeoutError expected');
      } catch (err) {
        err.name.should.equal('TimeoutError');
      }
      await clock.tickAsync(1500);
      _sinon2.default.assert.match(connection3.synchronized, false);
      await clock.tickAsync(2500);
      await new _promise2.default(res => setTimeout(res, 200));
      _sinon2.default.assert.match(connection3.synchronized, true);
    }).timeout(10000);

    it('should wait for retry time after per server 429 error', async () => {
      let requestTimestamp = 0;
      const sidByAccounts = {};

      fakeServer.io.removeAllListeners('connect');
      fakeServer.io.on('connect', socket => {
        server = socket;
        socket.emit('response', { type: 'response' });
        socket.removeAllListeners('request');
        //eslint-disable-next-line complexity
        socket.on('request', async data => {
          const sid = socket.id;
          if (data.type === 'subscribe') {
            if ((0, _values2.default)(sidByAccounts).filter(accountSID => accountSID === sid).length >= 2 && (requestTimestamp === 0 || Date.now() - 5 * 1000 < requestTimestamp)) {
              requestTimestamp = Date.now();
              await fakeServer.emitError(socket, data, 2, 5);
            } else {
              sidByAccounts[data.accountId] = sid;
              await new _promise2.default(res => setTimeout(res, 200));
              await fakeServer.respond(socket, data);
              fakeServer.statusTasks.accountId = setInterval(() => fakeServer.emitStatus(socket, data.accountId), 100);
              await new _promise2.default(res => setTimeout(res, 50));
              await fakeServer.authenticate(socket, data);
            }
          } else if (data.type === 'synchronize') {
            await fakeServer.respond(socket, data);
            await fakeServer.syncAccount(socket, data);
          } else if (data.type === 'waitSynchronized') {
            await fakeServer.respond(socket, data);
          } else if (data.type === 'getAccountInformation') {
            await fakeServer.respondAccountInformation(socket, data);
          }
        });
      });
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      await connection.waitSynchronized({ timeoutInSeconds: 3 });
      const account2 = await api.metatraderAccountApi.getAccount('accountId2');
      const connection2 = account2.getStreamingConnection();
      await connection2.connect();
      await connection2.waitSynchronized({ timeoutInSeconds: 3 });
      const account3 = await api.metatraderAccountApi.getAccount('accountId3');
      const connection3 = account3.getStreamingConnection();
      await connection3.connect();
      connection3.waitSynchronized({ timeoutInSeconds: 5 });
      await clock.tickAsync(5000);
      sidByAccounts.accountId.should.equal(sidByAccounts.accountId2);
      sidByAccounts.accountId2.should.not.equal(sidByAccounts.accountId3);
      await clock.tickAsync(5000);
      const account4 = await api.metatraderAccountApi.getAccount('accountId4');
      const connection4 = account4.getStreamingConnection();
      await connection4.connect();
      await new _promise2.default(res => setTimeout(res, 50));
      clock.tickAsync(5000);
      await connection4.waitSynchronized({ timeoutInSeconds: 3 });
      sidByAccounts.accountId.should.equal(sidByAccounts.accountId4);
    }).timeout(10000);

    it('should reconnect after per server 429 error if connection has no subscribed accounts', async () => {
      const sids = [];

      fakeServer.io.removeAllListeners('connect');
      fakeServer.io.on('connect', socket => {
        server = socket;
        socket.emit('response', { type: 'response' });
        socket.removeAllListeners('request');
        //eslint-disable-next-line complexity
        socket.on('request', async data => {
          const sid = socket.id;
          if (data.type === 'subscribe') {
            sids.push(sid);
            if (sids.length === 1) {
              await fakeServer.emitError(socket, data, 2, 2);
            } else {
              await new _promise2.default(res => setTimeout(res, 200));
              await fakeServer.respond(socket, data);
              fakeServer.statusTasks.accountId = setInterval(() => fakeServer.emitStatus(socket, data.accountId), 100);
              await new _promise2.default(res => setTimeout(res, 50));
              await fakeServer.authenticate(socket, data);
            }
          } else if (data.type === 'synchronize') {
            await fakeServer.respond(socket, data);
            await fakeServer.syncAccount(socket, data);
          } else if (data.type === 'waitSynchronized') {
            await fakeServer.respond(socket, data);
          } else if (data.type === 'getAccountInformation') {
            await fakeServer.respondAccountInformation(socket, data);
          }
        });
      });
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      await new _promise2.default(res => setTimeout(res, 50));
      clock.tickAsync(5000);
      await connection.waitSynchronized({ timeoutInSeconds: 5 });
      sids[0].should.not.equal(sids[1]);
    }).timeout(10000);

    it('should free a subscribe slot on unsubscribe after per server 429 error', async () => {
      const sidByAccounts = {};

      fakeServer.io.removeAllListeners('connect');
      fakeServer.io.on('connect', socket => {
        server = socket;
        socket.emit('response', { type: 'response' });
        socket.removeAllListeners('request');
        //eslint-disable-next-line complexity
        socket.on('request', async data => {
          const sid = socket.id;
          if (data.type === 'subscribe') {
            if ((0, _values2.default)(sidByAccounts).filter(accountSID => accountSID === sid).length >= 2) {
              await fakeServer.emitError(socket, data, 2, 200);
            } else {
              sidByAccounts[data.accountId] = sid;
              await new _promise2.default(res => setTimeout(res, 200));
              await fakeServer.respond(socket, data);
              fakeServer.statusTasks.accountId = setInterval(() => fakeServer.emitStatus(socket, data.accountId), 100);
              await new _promise2.default(res => setTimeout(res, 50));
              await fakeServer.authenticate(socket, data);
            }
          } else if (data.type === 'synchronize') {
            await fakeServer.respond(socket, data);
            await fakeServer.syncAccount(socket, data);
          } else if (data.type === 'waitSynchronized') {
            await fakeServer.respond(socket, data);
          } else if (data.type === 'getAccountInformation') {
            await fakeServer.respondAccountInformation(socket, data);
          } else if (data.type === 'unsubscribe') {
            delete sidByAccounts[data.accountId];
            await fakeServer.respond(socket, data);
          }
        });
      });
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      await connection.waitSynchronized({ timeoutInSeconds: 3 });
      const account2 = await api.metatraderAccountApi.getAccount('accountId2');
      const connection2 = account2.getStreamingConnection();
      await connection2.connect();
      await connection2.waitSynchronized({ timeoutInSeconds: 3 });
      const account3 = await api.metatraderAccountApi.getAccount('accountId3');
      const connection3 = account3.getStreamingConnection();
      await connection3.connect();
      connection3.waitSynchronized({ timeoutInSeconds: 5 });
      await clock.tickAsync(5000);
      sidByAccounts.accountId.should.equal(sidByAccounts.accountId2);
      sidByAccounts.accountId2.should.not.equal(sidByAccounts.accountId3);
      await connection2.close();
      const account4 = await api.metatraderAccountApi.getAccount('accountId4');
      const connection4 = account4.getStreamingConnection();
      await connection4.connect();
      await new _promise2.default(res => setTimeout(res, 50));
      clock.tickAsync(5000);
      await connection4.waitSynchronized({ timeoutInSeconds: 3 });
      sidByAccounts.accountId.should.equal(sidByAccounts.accountId4);
    }).timeout(10000);

    it('should wait for retry time after per server per user 429 error', async () => {
      let requestTimestamp = 0;
      const sidByAccounts = {};

      fakeServer.io.removeAllListeners('connect');
      fakeServer.io.on('connect', socket => {
        server = socket;
        socket.emit('response', { type: 'response' });
        socket.removeAllListeners('request');
        //eslint-disable-next-line complexity
        socket.on('request', async data => {
          const sid = socket.id;
          if (data.type === 'subscribe') {
            if ((0, _values2.default)(sidByAccounts).filter(accountSID => accountSID === sid).length >= 2 && (requestTimestamp === 0 || Date.now() - 2 * 1000 < requestTimestamp)) {
              requestTimestamp = Date.now();
              await fakeServer.emitError(socket, data, 0, 2);
            } else {
              sidByAccounts[data.accountId] = sid;
              await new _promise2.default(res => setTimeout(res, 200));
              await fakeServer.respond(socket, data);
              fakeServer.statusTasks.accountId = setInterval(() => fakeServer.emitStatus(socket, data.accountId), 100);
              await new _promise2.default(res => setTimeout(res, 50));
              await fakeServer.authenticate(socket, data);
            }
          } else if (data.type === 'synchronize') {
            await fakeServer.respond(socket, data);
            await fakeServer.syncAccount(socket, data);
          } else if (data.type === 'waitSynchronized') {
            await fakeServer.respond(socket, data);
          } else if (data.type === 'getAccountInformation') {
            await fakeServer.respondAccountInformation(socket, data);
          } else if (data.type === 'unsubscribe') {
            delete sidByAccounts[data.accountId];
            await fakeServer.respond(socket, data);
          }
        });
      });
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      await new _promise2.default(res => setTimeout(res, 50));
      clock.tickAsync(5000);
      await connection.waitSynchronized({ timeoutInSeconds: 5 });
      const account2 = await api.metatraderAccountApi.getAccount('accountId2');
      const connection2 = account2.getStreamingConnection();
      await connection2.connect();
      await new _promise2.default(res => setTimeout(res, 50));
      clock.tickAsync(5000);
      await connection2.waitSynchronized({ timeoutInSeconds: 5 });
      const account3 = await api.metatraderAccountApi.getAccount('accountId3');
      const connection3 = account3.getStreamingConnection();
      await connection3.connect();
      await new _promise2.default(res => setTimeout(res, 50));
      clock.tickAsync(5000);
      await connection3.waitSynchronized({ timeoutInSeconds: 5 });
      sidByAccounts.accountId.should.equal(sidByAccounts.accountId2);
      sidByAccounts.accountId2.should.not.equal(sidByAccounts.accountId3);
      const account4 = await api.metatraderAccountApi.getAccount('accountId4');
      const connection4 = account4.getStreamingConnection();
      await connection4.connect();
      await new _promise2.default(res => setTimeout(res, 50));
      clock.tickAsync(5000);
      await connection4.waitSynchronized({ timeoutInSeconds: 5 });
      sidByAccounts.accountId.should.not.equal(sidByAccounts.accountId4);
      await connection2.close();
      const account5 = await api.metatraderAccountApi.getAccount('accountId5');
      const connection5 = account5.getStreamingConnection();
      await connection5.connect();
      await new _promise2.default(res => setTimeout(res, 50));
      clock.tickAsync(5000);
      await connection5.waitSynchronized({ timeoutInSeconds: 5 });
      sidByAccounts.accountId.should.equal(sidByAccounts.accountId5);
    }).timeout(10000);

    it('should attempt to resubscribe on disconnected packet', async () => {
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      await new _promise2.default(res => setTimeout(res, 50));
      clock.tickAsync(5000);
      await connection.waitSynchronized({ timeoutInSeconds: 3 });
      (connection.synchronized && connection.terminalState.connected && connection.terminalState.connectedToBroker).should.equal(true);
      fakeServer.deleteStatusTask('accountId');
      await server.emit('synchronization', { type: 'disconnected', accountId: 'accountId',
        host: 'ps-mpa-0', instanceIndex: 0 });
      await new _promise2.default(res => setTimeout(res, 50));
      await clock.tickAsync(200);
      await new _promise2.default(res => setTimeout(res, 50));
      connection.synchronized.should.equal(false);
      connection.terminalState.connected.should.equal(false);
      connection.terminalState.connectedToBroker.should.equal(false);
      await clock.tickAsync(5000);
      (connection.synchronized && connection.terminalState.connected && connection.terminalState.connectedToBroker).should.equal(true);
    });

    it('should handle multiple streams in one instance number', async () => {
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      clock.tickAsync(5000);
      await connection.waitSynchronized({ timeoutInSeconds: 10 });
      let subscribeCalled = false;

      fakeServer.enableSync = socket => {
        socket.removeAllListeners('request');
        //eslint-disable-next-line complexity
        socket.on('request', async data => {
          if (data.type === 'subscribe') {
            subscribeCalled = true;
          } else if (data.type === 'synchronize') {
            await fakeServer.respond(socket, data);
            await fakeServer.syncAccount(socket, data, 'ps-mpa-1');
          } else if (data.type === 'waitSynchronized') {
            await fakeServer.respond(socket, data);
          } else if (data.type === 'getAccountInformation') {
            await fakeServer.respondAccountInformation(socket, data);
          }
        });
      };
      fakeServer.enableSync(server);
      const statusTask = setInterval(() => fakeServer.emitStatus(server, 'accountId', 'ps-mpa-1'), 100);
      await fakeServer.authenticate(server, { accountId: 'accountId' }, 'ps-mpa-1');
      await new _promise2.default(res => setTimeout(res, 50));
      await clock.tickAsync(20000);
      fakeServer.deleteStatusTask('accountId');
      await server.emit('synchronization', { type: 'disconnected', accountId: 'accountId',
        host: 'ps-mpa-0', instanceIndex: 0 });
      await new _promise2.default(res => setTimeout(res, 50));
      await clock.tickAsync(10000);
      (connection.synchronized && connection.terminalState.connected && connection.terminalState.connectedToBroker).should.equal(true);
      subscribeCalled.should.equal(false);
      await server.emit('synchronization', { type: 'disconnected', accountId: 'accountId',
        host: 'ps-mpa-1', instanceIndex: 0 });
      clearInterval(statusTask);
      await new _promise2.default(res => setTimeout(res, 50));
      connection.synchronized.should.equal(false);
      connection.terminalState.connected.should.equal(false);
      connection.terminalState.connectedToBroker.should.equal(false);
    }).timeout(10000);

    it('should not resubscribe if multiple streams and one timed out', async () => {
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      await new _promise2.default(res => setTimeout(res, 50));
      clock.tickAsync(5000);
      await connection.waitSynchronized({ timeoutInSeconds: 10 });
      let subscribeCalled = false;

      fakeServer.enableSync = socket => {
        socket.removeAllListeners('request');
        //eslint-disable-next-line complexity
        socket.on('request', async data => {
          if (data.type === 'subscribe') {
            subscribeCalled = true;
          } else if (data.type === 'synchronize') {
            await fakeServer.respond(socket, data);
            await fakeServer.syncAccount(socket, data, 'ps-mpa-1');
          } else if (data.type === 'waitSynchronized') {
            await fakeServer.respond(socket, data);
          } else if (data.type === 'getAccountInformation') {
            await fakeServer.respondAccountInformation(socket, data);
          }
        });
      };
      fakeServer.enableSync(server);
      const statusTask = setInterval(() => fakeServer.emitStatus(server, 'accountId', 'ps-mpa-1'), 120);
      await fakeServer.authenticate(server, { accountId: 'accountId' }, 'ps-mpa-1');
      await new _promise2.default(res => setTimeout(res, 50));
      await clock.tickAsync(20000);
      fakeServer.deleteStatusTask('accountId');
      await clock.tickAsync(55000);
      await new _promise2.default(res => setTimeout(res, 50));
      (connection.synchronized && connection.terminalState.connected && connection.terminalState.connectedToBroker).should.equal(true);
      subscribeCalled.should.equal(false);
      clearInterval(statusTask);
      await new _promise2.default(res => setTimeout(res, 50));
      await clock.tickAsync(65000);
      connection.synchronized.should.equal(false);
      connection.terminalState.connected.should.equal(false);
      connection.terminalState.connectedToBroker.should.equal(false);
      subscribeCalled.should.equal(true);
    }).timeout(10000);

    it('should not synchronize if connection is closed', async () => {
      let synchronizeCounter = 0;
      fakeServer.io.removeAllListeners('connect');
      fakeServer.io.on('connect', socket => {
        server = socket;
        socket.emit('response', { type: 'response' });
        socket.removeAllListeners('request');
        //eslint-disable-next-line complexity
        socket.on('request', async data => {
          if (data.type === 'subscribe') {
            await new _promise2.default(res => setTimeout(res, 200));
            await fakeServer.respond(socket, data);
            fakeServer.statusTasks.accountId = setInterval(() => fakeServer.emitStatus(socket, data.accountId), 100);
            await fakeServer.authenticate(socket, data);
          } else if (data.type === 'synchronize') {
            synchronizeCounter++;
            await fakeServer.respond(socket, data);
            await fakeServer.syncAccount(socket, data, 'ps-mpa-1');
          } else if (data.type === 'waitSynchronized') {
            await fakeServer.respond(socket, data);
          } else if (data.type === 'getAccountInformation') {
            await fakeServer.respondAccountInformation(socket, data);
          } else if (data.type === 'unsubscribe') {
            fakeServer.deleteStatusTask(data.accountId);
            await fakeServer.respond(socket, data);
          }
        });
      });
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      await new _promise2.default(res => setTimeout(res, 50));
      clock.tickAsync(5000);
      await connection.waitSynchronized({ timeoutInSeconds: 10 });
      _sinon2.default.assert.match(synchronizeCounter, 1);
      const account2 = await api.metatraderAccountApi.getAccount('accountId2');
      const connection2 = account2.getStreamingConnection();
      await connection2.connect();
      await new _promise2.default(res => setTimeout(res, 50));
      await connection2.close();
      try {
        clock.tickAsync(5000);
        await connection2.waitSynchronized({ timeoutInSeconds: 5 });
        throw new Error('TimeoutError extected');
      } catch (err) {
        err.name.should.equal('TimeoutError');
      }
      _sinon2.default.assert.match(synchronizeCounter, 1);
    }).timeout(10000);

    it('should not resubscribe after connection is closed', async () => {
      let subscribeCounter = 0;

      fakeServer.io.removeAllListeners('connect');
      fakeServer.io.on('connect', socket => {
        server = socket;
        socket.emit('response', { type: 'response' });
        socket.removeAllListeners('request');
        //eslint-disable-next-line complexity
        socket.on('request', async data => {
          if (data.type === 'subscribe') {
            subscribeCounter++;
            await new _promise2.default(res => setTimeout(res, 100));
            await fakeServer.respond(socket, data);
            fakeServer.deleteStatusTask(data.accountId);
            fakeServer.statusTasks.accountId = setInterval(() => fakeServer.emitStatus(socket, data.accountId), 100);
            await fakeServer.authenticate(socket, data);
          } else if (data.type === 'synchronize') {
            await fakeServer.respond(socket, data);
            await fakeServer.syncAccount(socket, data);
          } else if (data.type === 'waitSynchronized') {
            await fakeServer.respond(socket, data);
          } else if (data.type === 'getAccountInformation') {
            await fakeServer.respondAccountInformation(socket, data);
          } else if (data.type === 'unsubscribe') {
            fakeServer.deleteStatusTask(data.accountId);
            await fakeServer.respond(socket, data);
          }
        });
      });

      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      clock.tickAsync(5000);
      await connection.waitSynchronized({ timeoutInSeconds: 10 });
      (connection.synchronized && connection.terminalState.connected && connection.terminalState.connectedToBroker).should.equal(true);
      subscribeCounter.should.equal(1);
      await server.emit('synchronization', { type: 'disconnected', accountId: 'accountId',
        host: 'ps-mpa-0', instanceIndex: 0 });
      await new _promise2.default(res => setTimeout(res, 50));
      await clock.tickAsync(100000);
      await new _promise2.default(res => setTimeout(res, 50));
      subscribeCounter.should.be.above(1);
      const previousSubscribeCounter = subscribeCounter;
      (connection.synchronized && connection.terminalState.connected && connection.terminalState.connectedToBroker).should.equal(true);
      server.emit('synchronization', { type: 'disconnected', accountId: 'accountId',
        host: 'ps-mpa-0', instanceIndex: 0 });
      await connection.close();
      await new _promise2.default(res => setTimeout(res, 50));
      await clock.tickAsync(100000);
      await new _promise2.default(res => setTimeout(res, 50));
      _sinon2.default.assert.match(subscribeCounter, previousSubscribeCounter);
      connection.synchronized.should.equal(false);
      connection.terminalState.connected.should.equal(false);
      connection.terminalState.connectedToBroker.should.equal(false);
    }).timeout(10000);

    it('should not resubscribe on timeout if connection is closed', async () => {
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      clock.tickAsync(5000);
      await connection.waitSynchronized({ timeoutInSeconds: 10 });
      fakeServer.deleteStatusTask('accountId');
      connection.synchronized.should.equal(true);
      await connection.close();
      await new _promise2.default(res => setTimeout(res, 100));
      await clock.tickAsync(62000);
      connection.synchronized.should.equal(false);
    });

    it('should not send multiple subscribe requests if status arrives faster than subscribe', async () => {
      let subscribeCounter = 0;
      const account = await api.metatraderAccountApi.getAccount('accountId');
      connection = account.getStreamingConnection();
      await connection.connect();
      clock.tickAsync(5000);
      await connection.waitSynchronized({ timeoutInSeconds: 10 });
      fakeServer.disableSync();
      fakeServer.deleteStatusTask('accountId');
      await new _promise2.default(res => setTimeout(res, 50));
      await clock.tickAsync(120000);
      connection.synchronized.should.equal(false);
      connection.terminalState.connected.should.equal(false);
      connection.terminalState.connectedToBroker.should.equal(false);

      server.removeAllListeners('request');
      //eslint-disable-next-line complexity
      server.on('request', async data => {
        if (data.type === 'subscribe') {
          subscribeCounter++;
          await new _promise2.default(res => setTimeout(res, 2800));
          await fakeServer.respond(server, data);
          fakeServer.deleteStatusTask(data.accountId);
          fakeServer.statusTasks.accountId = setInterval(() => fakeServer.emitStatus(server, data.accountId), 1000);
          await fakeServer.authenticate(server, data);
        } else if (data.type === 'synchronize') {
          await fakeServer.respond(server, data);
          await fakeServer.syncAccount(server, data);
        } else if (data.type === 'waitSynchronized') {
          await fakeServer.respond(server, data);
        } else if (data.type === 'getAccountInformation') {
          await fakeServer.respondAccountInformation(server, data);
        } else if (data.type === 'unsubscribe') {
          fakeServer.deleteStatusTask(data.accountId);
          await fakeServer.respond(server, data);
        }
      });
      fakeServer.statusTasks.accountId = setInterval(() => fakeServer.emitStatus(server, 'accountId'), 1000);
      await new _promise2.default(res => setTimeout(res, 100));
      await clock.tickAsync(200000);
      await new _promise2.default(res => setTimeout(res, 100));
      await clock.tickAsync(5000);
      (connection.synchronized && connection.terminalState.connected && connection.terminalState.connectedToBroker).should.equal(true);
      _sinon2.default.assert.match(subscribeCounter, 1);
    }).timeout(10000);
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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