'use strict';

const log4js = require('@log4js-node/log4js-api');

let log4jsEnabled = false;

/**
 * Manages loggers of the entire sdk
 */
class LoggerManager {

  /**
   * Enables using Log4js logger with extended log levels for debugging instead of
   * console.* functions. Note that log4js configuration performed by the user.
   */
  static useLog4js() {
    log4jsEnabled = true;
  }

  /**
   * Creates a new logger for specified category
   * @param {String} category Logger category
   * @return {Logger} Created logger
   */
  static getLogger(category) {
    if (log4jsEnabled) {
      let logger = log4js.getLogger(category);
      if (logger._log) {
        let originalLog = logger._log.bind(logger);
        logger._log = function (level, data) {
          if (typeof data[0] === 'function') {
            data[0] = data[0]();
          }
          originalLog(level, data);
        };
      }
      return logger;
    } else {
      return new NativeLogger();
    }
  }
}

/**
 * Native logger that uses console.* functions
 */
class NativeLogger {

  /**
   * Supposed to log messages of trace level
   * @param {String|Function} message Message or message generator to log
   * @param {*[]} args Additional args to message
   */
  trace(message, ...args) {}
  // this logger does not print trace messages


  /**
   * Supposed to log messages of debug level
   * @param {String|Function} message Message or message generator to log
   * @param {*[]} args Additional args to message
   */
  debug(message, ...args) {}
  // this logger does not print debug messages


  /**
   * Prints a message with console.log
   * @param {String|Function} message Message or message generator to log
   * @param {*[]} args Additional args to message
   */
  info(message, ...args) {
    this._log('log', message, args);
  }

  /**
   * Prints a message with console.warn
   * @param {String|Function} message Message or message generator to log
   * @param {*[]} args Additional args to message
   */
  warn(message, ...args) {
    this._log('warn', message, args);
  }

  /**
   * Prints a message with console.error
   * @param {String|Function} message Message or message generator to log
   * @param {*[]} args Additional args to message
   */
  error(message, ...args) {
    this._log('error', message, args);
  }

  /**
   * Prints a message with console.error
   * @param {String|Function} message Message or message generator to log
   * @param {*[]} args Additional args to message
   */
  fatal(message, ...args) {
    this._log('error', message, args);
  }

  _log(level, message, args) {
    if (typeof message === 'function') {
      message = message();
    }
    console[level](`[${new Date().toISOString()}] ${message}`, ...args);
  }
}

module.exports = LoggerManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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