'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _streamingMetaApiConnection = require('./streamingMetaApiConnection');

var _streamingMetaApiConnection2 = _interopRequireDefault(_streamingMetaApiConnection);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Manages account connections
 */
class ConnectionRegistry {

  /**
   * Constructs a MetaTrader connection registry instance
   * @param {MetaApiWebsocketClient} metaApiWebsocketClient MetaApi websocket client
   * @param {String} application application id
   * @param {String} refreshSubscriptionsOpts subscriptions refresh options
   */
  constructor(metaApiWebsocketClient, application = 'MetaApi', refreshSubscriptionsOpts) {
    refreshSubscriptionsOpts = refreshSubscriptionsOpts || {};
    this._metaApiWebsocketClient = metaApiWebsocketClient;
    this._application = application;
    this._refreshSubscriptionsOpts = refreshSubscriptionsOpts;
    this._connections = {};
    this._connectionLocks = {};
  }

  /**
   * Creates and returns a new account connection if doesnt exist, otherwise returns old
   * @param {MetatraderAccount} account MetaTrader account id to connect to
   * @param {HistoryStorage} historyStorage terminal history storage
   * @param {Date} [historyStartTime] history start time
   * @return {StreamingMetaApiConnection} streaming metaapi connection
   */
  connect(account, historyStorage, historyStartTime) {
    if (this._connections[account.id]) {
      return this._connections[account.id];
    }
    this._connections[account.id] = new _streamingMetaApiConnection2.default(this._metaApiWebsocketClient, account, historyStorage, this, historyStartTime, this._refreshSubscriptionsOpts);
    return this._connections[account.id];
  }

  /**
   * Removes an account from registry
   * @param {string} accountId MetaTrader account id to remove
   */
  remove(accountId) {
    if (this._connections[accountId]) {
      delete this._connections[accountId];
    }
  }

  /**
   * Returns application type
   * @return {String} application type
   */
  get application() {
    return this._application;
  }
}
exports.default = ConnectionRegistry;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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