'use strict';

const FileManager = require('./fileManager');
const { openDB, deleteDB } = require('idb');
const LoggerManager = require('../../logger');

/**
 * History storage manager which saves and loads history to IndexedDB
 */
module.exports = class BrowserHistoryManager extends FileManager {

  /**
   * Constructs the history file manager instance
   * @param {String} accountId MetaApi account id
   * @param {String} application MetaApi application id
   * @param {HistoryStorage} historyStorage history storage
   */
  constructor(accountId, application, historyStorage) {
    super(accountId, application, historyStorage);
    this._logger = LoggerManager.getLogger('BrowserHistoryManager');
  }

  /**
   * Retrieves history from IndexedDB
   * @returns {Object} object with deals and historyOrders
   */
  // eslint-disable-next-line complexity
  async getHistoryFromDisk() {
    try {
      const history = {
        deals: [],
        historyOrders: [],
        lastDealTimeByInstanceIndex: {},
        lastHistoryOrderTimeByInstanceIndex: {}
      };
      const db = await this._getDatabase();
      const deals = await db.get('deals', this._accountId + '-' + this._application);
      history.deals = deals && deals.items || [];
      history.lastDealTimeByInstanceIndex = deals && deals.lastDealTimeByInstanceIndex || {};
      const historyOrders = await db.get('historyOrders', this._accountId + '-' + this._application);
      history.historyOrders = historyOrders && historyOrders.items || [];
      history.lastHistoryOrderTimeByInstanceIndex = historyOrders && historyOrders.lastHistoryOrderTimeByInstanceIndex || {};
      db.close();
      return history;
    } catch (err) {
      this._logger.error(`${this._accountId}: Failed to get history from IndexedDB`, err);
    }
  }

  /**
   * Saves history items to IndexedDB
   */
  async updateDiskStorage() {
    try {
      const db = await this._getDatabase();
      await db.put('deals', { accountIdAndApplication: this._accountId + '-' + this._application,
        items: this._historyStorage.deals,
        lastDealTimeByInstanceIndex: this._historyStorage.lastDealTimeByInstanceIndex });
      await db.put('historyOrders', { accountIdAndApplication: this._accountId + '-' + this._application,
        items: this._historyStorage.historyOrders,
        lastHistoryOrderTimeByInstanceIndex: this._historyStorage.lastHistoryOrderTimeByInstanceIndex });
      db.close();
    } catch (err) {
      this._logger.error(`${this._accountId}: Failed to save history into IndexedDB`, err);
    }
  }

  /**
   * Deletes storage files from disk
   */
  async deleteStorageFromDisk() {
    try {
      const db = await this._getDatabase();
      await db.delete('deals', this._accountId + '-' + this._application);
      await db.delete('historyOrders', this._accountId + '-' + this._application);
      db.close();
    } catch (err) {
      this._logger.error(`${this._accountId}: Failed to delete history from IndexedDB`, err);
    }
  }

  /**
   * Opens an IndexedDB database and verifies its structure, recreates if structure is invalid, then returns
   * @returns {IndexedDB} indexed db
   */
  async _getDatabase() {
    const keyPath = 'accountIdAndApplication';
    const db = await openDB('metaapi', 1, {
      upgrade(database, oldVersion, newVersion, transaction) {
        if (!database.objectStoreNames.contains('deals')) {
          database.createObjectStore('deals', { keyPath });
        }
        if (!database.objectStoreNames.contains('historyOrders')) {
          database.createObjectStore('historyOrders', { keyPath });
        }
      }
    });
    if (!db.objectStoreNames.contains('deals') || !db.objectStoreNames.contains('historyOrders')) {
      db.close();
      await deleteDB('metaapi');
      return await this._getDatabase();
    } else {
      const tr = db.transaction(['deals', 'historyOrders'], 'readonly');
      if (tr.objectStore('deals').keyPath !== keyPath || tr.objectStore('historyOrders').keyPath !== keyPath) {
        db.close();
        await deleteDB('metaapi');
        return await this._getDatabase();
      } else {
        return db;
      }
    }
  }

};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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