'use strict';

var _promise = require('babel-runtime/core-js/promise');

var _promise2 = _interopRequireDefault(_promise);

var _should = require('should');

var _should2 = _interopRequireDefault(_should);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _rpcMetaApiConnection = require('./rpcMetaApiConnection');

var _rpcMetaApiConnection2 = _interopRequireDefault(_rpcMetaApiConnection);

var _notSynchronizedError = require('../clients/metaApi/notSynchronizedError');

var _notSynchronizedError2 = _interopRequireDefault(_notSynchronizedError);

var _index = require('./historyFileManager/index');

var _index2 = _interopRequireDefault(_index);

var _timeoutError = require('../clients/timeoutError');

var _timeoutError2 = _interopRequireDefault(_timeoutError);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * @test {MetaApiConnection}
 */
// eslint-disable-next-line max-statements
describe('RpcMetaApiConnection', () => {

  let sandbox;
  let api;
  let account;
  let clock;
  let client = {
    getAccountInformation: () => {},
    getPositions: () => {},
    getPosition: () => {},
    getOrders: () => {},
    getOrder: () => {},
    getHistoryOrdersByTicket: () => {},
    getHistoryOrdersByPosition: () => {},
    getHistoryOrdersByTimeRange: () => {},
    getDealsByTicket: () => {},
    getDealsByPosition: () => {},
    getDealsByTimeRange: () => {},
    removeHistory: () => {},
    removeApplication: () => {},
    trade: () => {},
    reconnect: () => {},
    getSymbols: () => {},
    getSymbolSpecification: () => {},
    getSymbolPrice: () => {},
    getCandle: () => {},
    getTick: () => {},
    getBook: () => {},
    waitSynchronized: () => {}
  };

  before(() => {
    sandbox = _sinon2.default.createSandbox();
  });

  beforeEach(() => {
    account = {
      id: 'accountId',
      state: 'DEPLOYED',
      reload: () => {}
    };
    sandbox.stub(_index2.default.prototype, 'startUpdateJob').returns();
    api = new _rpcMetaApiConnection2.default(client, account);
    clock = _sinon2.default.useFakeTimers({
      shouldAdvanceTime: true
    });
  });

  afterEach(() => {
    clock.restore();
    sandbox.restore();
  });

  /**
   * @test {MetaApiConnection#waitSynchronized}
   */
  it('should wait until RPC application is synchronized', async () => {
    sandbox.stub(client, 'waitSynchronized').onFirstCall().rejects(new _timeoutError2.default('test')).onSecondCall().rejects(new _timeoutError2.default('test')).onThirdCall().resolves('response');
  });

  /**
   * @test {MetaApiConnection#waitSynchronized}
   */
  it('should time out waiting for synchronization', async () => {
    sandbox.stub(client, 'waitSynchronized').callsFake(async () => {
      await new _promise2.default(res => setTimeout(res, 100));
      throw new _timeoutError2.default('test');
    });
    try {
      await api.waitSynchronized(0.09);
      throw new Error('TimeoutError extected');
    } catch (err) {
      err.name.should.equal('TimeoutError');
    }
  });

  /**
   * @test {MetaApiConnection#getAccountInformation}
   */
  it('should retrieve account information', async () => {
    let accountInformation = {
      broker: 'True ECN Trading Ltd',
      currency: 'USD',
      server: 'ICMarketsSC-Demo',
      balance: 7319.9,
      equity: 7306.649913200001,
      margin: 184.1,
      freeMargin: 7120.22,
      leverage: 100,
      marginLevel: 3967.58283542
    };
    sandbox.stub(client, 'getAccountInformation').resolves(accountInformation);
    let actual = await api.getAccountInformation();
    actual.should.match(accountInformation);
    _sinon2.default.assert.calledWith(client.getAccountInformation, 'accountId');
  });

  /**
   * @test {MetaApiConnection#getPositions}
   */
  it('should retrieve positions', async () => {
    let positions = [{
      id: '46214692',
      type: 'POSITION_TYPE_BUY',
      symbol: 'GBPUSD',
      magic: 1000,
      time: new Date('2020-04-15T02:45:06.521Z'),
      updateTime: new Date('2020-04-15T02:45:06.521Z'),
      openPrice: 1.26101,
      currentPrice: 1.24883,
      currentTickValue: 1,
      volume: 0.07,
      swap: 0,
      profit: -85.25999999999966,
      commission: -0.25,
      clientId: 'TE_GBPUSD_7hyINWqAlE',
      stopLoss: 1.17721,
      unrealizedProfit: -85.25999999999901,
      realizedProfit: -6.536993168992922e-13
    }];
    sandbox.stub(client, 'getPositions').resolves(positions);
    let actual = await api.getPositions();
    actual.should.match(positions);
    _sinon2.default.assert.calledWith(client.getPositions, 'accountId');
  });

  /**
   * @test {MetaApiConnection#getPosition}
   */
  it('should retrieve position by id', async () => {
    let position = {
      id: '46214692',
      type: 'POSITION_TYPE_BUY',
      symbol: 'GBPUSD',
      magic: 1000,
      time: new Date('2020-04-15T02:45:06.521Z'),
      updateTime: new Date('2020-04-15T02:45:06.521Z'),
      openPrice: 1.26101,
      currentPrice: 1.24883,
      currentTickValue: 1,
      volume: 0.07,
      swap: 0,
      profit: -85.25999999999966,
      commission: -0.25,
      clientId: 'TE_GBPUSD_7hyINWqAlE',
      stopLoss: 1.17721,
      unrealizedProfit: -85.25999999999901,
      realizedProfit: -6.536993168992922e-13
    };
    sandbox.stub(client, 'getPosition').resolves(position);
    let actual = await api.getPosition('46214692');
    actual.should.match(position);
    _sinon2.default.assert.calledWith(client.getPosition, 'accountId', '46214692');
  });

  /**
   * @test {MetaApiConnection#getOrders}
   */
  it('should retrieve orders', async () => {
    let orders = [{
      id: '46871284',
      type: 'ORDER_TYPE_BUY_LIMIT',
      state: 'ORDER_STATE_PLACED',
      symbol: 'AUDNZD',
      magic: 123456,
      platform: 'mt5',
      time: new Date('2020-04-20T08:38:58.270Z'),
      openPrice: 1.03,
      currentPrice: 1.05206,
      volume: 0.01,
      currentVolume: 0.01,
      comment: 'COMMENT2'
    }];
    sandbox.stub(client, 'getOrders').resolves(orders);
    let actual = await api.getOrders();
    actual.should.match(orders);
    _sinon2.default.assert.calledWith(client.getOrders, 'accountId');
  });

  /**
   * @test {MetaApiConnection#getOrder}
   */
  it('should retrieve order by id', async () => {
    let order = {
      id: '46871284',
      type: 'ORDER_TYPE_BUY_LIMIT',
      state: 'ORDER_STATE_PLACED',
      symbol: 'AUDNZD',
      magic: 123456,
      platform: 'mt5',
      time: new Date('2020-04-20T08:38:58.270Z'),
      openPrice: 1.03,
      currentPrice: 1.05206,
      volume: 0.01,
      currentVolume: 0.01,
      comment: 'COMMENT2'
    };
    sandbox.stub(client, 'getOrder').resolves(order);
    let actual = await api.getOrder('46871284');
    actual.should.match(order);
    _sinon2.default.assert.calledWith(client.getOrder, 'accountId', '46871284');
  });

  /**
   * @test {MetaApiConnection#getHistoryOrdersByTicket}
   */
  it('should retrieve history orders by ticket', async () => {
    let historyOrders = {
      historyOrders: [{
        clientId: 'TE_GBPUSD_7hyINWqAlE',
        currentPrice: 1.261,
        currentVolume: 0,
        doneTime: new Date('2020-04-15T02:45:06.521Z'),
        id: '46214692',
        magic: 1000,
        platform: 'mt5',
        positionId: '46214692',
        state: 'ORDER_STATE_FILLED',
        symbol: 'GBPUSD',
        time: new Date('2020-04-15T02:45:06.260Z'),
        type: 'ORDER_TYPE_BUY',
        volume: 0.07
      }],
      synchronizing: false
    };
    sandbox.stub(client, 'getHistoryOrdersByTicket').resolves(historyOrders);
    let actual = await api.getHistoryOrdersByTicket('46214692');
    actual.should.match(historyOrders);
    _sinon2.default.assert.calledWith(client.getHistoryOrdersByTicket, 'accountId', '46214692');
  });

  /**
   * @test {MetaApiConnection#getHistoryOrdersByPosition}
   */
  it('should retrieve history orders by position', async () => {
    let historyOrders = {
      historyOrders: [{
        clientId: 'TE_GBPUSD_7hyINWqAlE',
        currentPrice: 1.261,
        currentVolume: 0,
        doneTime: new Date('2020-04-15T02:45:06.521Z'),
        id: '46214692',
        magic: 1000,
        platform: 'mt5',
        positionId: '46214692',
        state: 'ORDER_STATE_FILLED',
        symbol: 'GBPUSD',
        time: new Date('2020-04-15T02:45:06.260Z'),
        type: 'ORDER_TYPE_BUY',
        volume: 0.07
      }],
      synchronizing: false
    };
    sandbox.stub(client, 'getHistoryOrdersByPosition').resolves(historyOrders);
    let actual = await api.getHistoryOrdersByPosition('46214692');
    actual.should.match(historyOrders);
    _sinon2.default.assert.calledWith(client.getHistoryOrdersByPosition, 'accountId', '46214692');
  });

  /**
   * @test {MetaApiConnection#getHistoryOrdersByTimeRange}
   */
  it('should retrieve history orders by time range', async () => {
    let historyOrders = {
      historyOrders: [{
        clientId: 'TE_GBPUSD_7hyINWqAlE',
        currentPrice: 1.261,
        currentVolume: 0,
        doneTime: new Date('2020-04-15T02:45:06.521Z'),
        id: '46214692',
        magic: 1000,
        platform: 'mt5',
        positionId: '46214692',
        state: 'ORDER_STATE_FILLED',
        symbol: 'GBPUSD',
        time: new Date('2020-04-15T02:45:06.260Z'),
        type: 'ORDER_TYPE_BUY',
        volume: 0.07
      }],
      synchronizing: false
    };
    sandbox.stub(client, 'getHistoryOrdersByTimeRange').resolves(historyOrders);
    let startTime = new Date(Date.now() - 1000);
    let endTime = new Date();
    let actual = await api.getHistoryOrdersByTimeRange(startTime, endTime, 1, 100);
    actual.should.match(historyOrders);
    _sinon2.default.assert.calledWith(client.getHistoryOrdersByTimeRange, 'accountId', startTime, endTime, 1, 100);
  });

  /**
   * @test {MetaApiConnection#getDealsByTicket}
   */
  it('should retrieve history deals by ticket', async () => {
    let deals = {
      deals: [{
        clientId: 'TE_GBPUSD_7hyINWqAlE',
        commission: -0.25,
        entryType: 'DEAL_ENTRY_IN',
        id: '33230099',
        magic: 1000,
        platform: 'mt5',
        orderId: '46214692',
        positionId: '46214692',
        price: 1.26101,
        profit: 0,
        swap: 0,
        symbol: 'GBPUSD',
        time: new Date('2020-04-15T02:45:06.521Z'),
        type: 'DEAL_TYPE_BUY',
        volume: 0.07
      }],
      synchronizing: false
    };
    sandbox.stub(client, 'getDealsByTicket').resolves(deals);
    let actual = await api.getDealsByTicket('46214692');
    actual.should.match(deals);
    _sinon2.default.assert.calledWith(client.getDealsByTicket, 'accountId', '46214692');
  });

  /**
   * @test {MetaApiConnection#getDealsByPosition}
   */
  it('should retrieve history deals by position', async () => {
    let deals = {
      deals: [{
        clientId: 'TE_GBPUSD_7hyINWqAlE',
        commission: -0.25,
        entryType: 'DEAL_ENTRY_IN',
        id: '33230099',
        magic: 1000,
        platform: 'mt5',
        orderId: '46214692',
        positionId: '46214692',
        price: 1.26101,
        profit: 0,
        swap: 0,
        symbol: 'GBPUSD',
        time: new Date('2020-04-15T02:45:06.521Z'),
        type: 'DEAL_TYPE_BUY',
        volume: 0.07
      }],
      synchronizing: false
    };
    sandbox.stub(client, 'getDealsByPosition').resolves(deals);
    let actual = await api.getDealsByPosition('46214692');
    actual.should.match(deals);
    _sinon2.default.assert.calledWith(client.getDealsByPosition, 'accountId', '46214692');
  });

  /**
   * @test {MetaApiConnection#getDealsByTimeRange}
   */
  it('should retrieve history deals by time range', async () => {
    let deals = {
      deals: [{
        clientId: 'TE_GBPUSD_7hyINWqAlE',
        commission: -0.25,
        entryType: 'DEAL_ENTRY_IN',
        id: '33230099',
        magic: 1000,
        platform: 'mt5',
        orderId: '46214692',
        positionId: '46214692',
        price: 1.26101,
        profit: 0,
        swap: 0,
        symbol: 'GBPUSD',
        time: new Date('2020-04-15T02:45:06.521Z'),
        type: 'DEAL_TYPE_BUY',
        volume: 0.07
      }],
      synchronizing: false
    };
    sandbox.stub(client, 'getDealsByTimeRange').resolves(deals);
    let startTime = new Date(Date.now() - 1000);
    let endTime = new Date();
    let actual = await api.getDealsByTimeRange(startTime, endTime, 1, 100);
    actual.should.match(deals);
    _sinon2.default.assert.calledWith(client.getDealsByTimeRange, 'accountId', startTime, endTime, 1, 100);
  });

  /**
   * @test {MetaApiConnection#removeHistory}
   */
  it('should remove history', async () => {
    sandbox.stub(client, 'removeHistory').resolves();
    await api.removeHistory('app');
    _sinon2.default.assert.calledWith(client.removeHistory, 'accountId', 'app');
  });

  /**
   * @test {MetaApiConnection#createMarketBuyOrder}
   */
  it('should create market buy order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: 46870472
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.createMarketBuyOrder('GBPUSD', 0.07, 0.9, 2.0, { comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_TYPE_BUY', symbol: 'GBPUSD',
      volume: 0.07, stopLoss: 0.9, takeProfit: 2.0, comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' }), 'RPC');
  });

  /**
   * @test {MetaApiConnection#createMarketBuyOrder}
   */
  it('should create market buy order with relative SL/TP', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: 46870472
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.createMarketBuyOrder('GBPUSD', 0.07, { value: 0.1, units: 'RELATIVE_PRICE' }, { value: 2000, units: 'RELATIVE_POINTS' }, { comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_TYPE_BUY', symbol: 'GBPUSD',
      volume: 0.07, stopLoss: 0.1, stopLossUnits: 'RELATIVE_PRICE', takeProfit: 2000,
      takeProfitUnits: 'RELATIVE_POINTS', comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' }), 'RPC');
  });

  /**
   * @test {MetaApiConnection#createMarketSellOrder}
   */
  it('should create market sell order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: 46870472
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.createMarketSellOrder('GBPUSD', 0.07, 2.0, 0.9, { comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_TYPE_SELL', symbol: 'GBPUSD',
      volume: 0.07, stopLoss: 2.0, takeProfit: 0.9, comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' }), 'RPC');
  });

  /**
   * @test {MetaApiConnection#createLimitBuyOrder}
   */
  it('should create limit buy order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: 46870472
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.createLimitBuyOrder('GBPUSD', 0.07, 1.0, 0.9, 2.0, { comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_TYPE_BUY_LIMIT',
      symbol: 'GBPUSD', volume: 0.07, openPrice: 1.0, stopLoss: 0.9, takeProfit: 2.0, comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' }), 'RPC');
  });

  /**
   * @test {MetaApiConnection#createLimitSellOrder}
   */
  it('should create limit sell order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: 46870472
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.createLimitSellOrder('GBPUSD', 0.07, 1.5, 2.0, 0.9, { comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_TYPE_SELL_LIMIT',
      symbol: 'GBPUSD', volume: 0.07, openPrice: 1.5, stopLoss: 2.0, takeProfit: 0.9, comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' }), 'RPC');
  });

  /**
   * @test {MetaApiConnection#createStopBuyOrder}
   */
  it('should create stop buy order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: 46870472
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.createStopBuyOrder('GBPUSD', 0.07, 1.5, 0.9, 2.0, { comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_TYPE_BUY_STOP',
      symbol: 'GBPUSD', volume: 0.07, openPrice: 1.5, stopLoss: 0.9, takeProfit: 2.0, comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' }), 'RPC');
  });

  /**
   * @test {MetaApiConnection#createStopSellOrder}
   */
  it('should create stop sell order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: '46870472'
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.createStopSellOrder('GBPUSD', 0.07, 1.0, 2.0, 0.9, { comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_TYPE_SELL_STOP',
      symbol: 'GBPUSD', volume: 0.07, openPrice: 1.0, stopLoss: 2.0, takeProfit: 0.9, comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' }), 'RPC');
  });

  /**
   * @test {MetaApiConnection#createStopLimitBuyOrder}
   */
  it('should create stop limit buy order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: 46870472
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.createStopLimitBuyOrder('GBPUSD', 0.07, 1.5, 1.4, 0.9, 2.0, { comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_TYPE_BUY_STOP_LIMIT',
      symbol: 'GBPUSD', volume: 0.07, openPrice: 1.5, stopLimitPrice: 1.4, stopLoss: 0.9, takeProfit: 2.0,
      comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' }), 'RPC');
  });

  /**
   * @test {MetaApiConnection#createStopLimitSellOrder}
   */
  it('should create stop limit sell order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: '46870472'
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.createStopLimitSellOrder('GBPUSD', 0.07, 1.0, 1.1, 2.0, 0.9, { comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_TYPE_SELL_STOP_LIMIT',
      symbol: 'GBPUSD', volume: 0.07, openPrice: 1.0, stopLimitPrice: 1.1, stopLoss: 2.0, takeProfit: 0.9,
      comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' }), 'RPC');
  });

  /**
   * @test {MetaApiConnection#modifyPosition}
   */
  it('should modify position', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      positionId: '46870472'
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.modifyPosition('46870472', 2.0, 0.9);
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'POSITION_MODIFY',
      positionId: '46870472', stopLoss: 2.0, takeProfit: 0.9 }), 'RPC');
  });

  /**
   * @test {MetaApiConnection#closePositionPartially}
   */
  it('should close position partially', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      positionId: '46870472'
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.closePositionPartially('46870472', 0.9, { comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'POSITION_PARTIAL',
      positionId: '46870472', volume: 0.9, comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' }), 'RPC');
  });

  /**
   * @test {MetaApiConnection#closePosition}
   */
  it('should close position', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      positionId: '46870472'
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.closePosition('46870472', { comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'POSITION_CLOSE_ID',
      positionId: '46870472', comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' }), 'RPC');
  });

  /**
   * @test {MetaApiConnection#closeBy}
   */
  it('should close position by an opposite one', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      positionId: '46870472',
      closeByPositionId: '46870482'
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.closeBy('46870472', '46870482', { comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'POSITION_CLOSE_BY',
      positionId: '46870472', closeByPositionId: '46870482', comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' }), 'RPC');
  });

  /**
   * @test {MetaApiConnection#closePositionsBySymbol}
   */
  it('should close positions by symbol', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      positionId: '46870472'
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.closePositionsBySymbol('EURUSD', { comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'POSITIONS_CLOSE_SYMBOL',
      symbol: 'EURUSD', comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' }), 'RPC');
  });

  /**
   * @test {MetaApiConnection#modifyOrder}
   */
  it('should modify order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: '46870472'
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.modifyOrder('46870472', 1.0, 2.0, 0.9);
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_MODIFY', orderId: '46870472',
      openPrice: 1.0, stopLoss: 2.0, takeProfit: 0.9 }), 'RPC');
  });

  /**
   * @test {MetaApiConnection#cancelOrder}
   */
  it('should cancel order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: '46870472'
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.cancelOrder('46870472');
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_CANCEL', orderId: '46870472' }), 'RPC');
  });

  /**
   * @test {MetaApiConnection#getSymbols}
   */
  it('should retrieve symbols', async () => {
    let symbols = ['EURUSD'];
    sandbox.stub(client, 'getSymbols').resolves(symbols);
    let actual = await api.getSymbols();
    actual.should.match(symbols);
    _sinon2.default.assert.calledWith(client.getSymbols, 'accountId');
  });

  /**
   * @test {MetaApiConnection#getSymbolSpecification}
   */
  it('should retrieve symbol specification', async () => {
    let specification = {
      symbol: 'AUDNZD',
      tickSize: 0.00001,
      minVolume: 0.01,
      maxVolume: 100,
      volumeStep: 0.01
    };
    sandbox.stub(client, 'getSymbolSpecification').resolves(specification);
    let actual = await api.getSymbolSpecification('AUDNZD');
    actual.should.match(specification);
    _sinon2.default.assert.calledWith(client.getSymbolSpecification, 'accountId', 'AUDNZD');
  });

  /**
   * @test {MetaApiConnection#getSymbolPrice}
   */
  it('should retrieve symbol price', async () => {
    let price = {
      symbol: 'AUDNZD',
      bid: 1.05297,
      ask: 1.05309,
      profitTickValue: 0.59731,
      lossTickValue: 0.59736
    };
    sandbox.stub(client, 'getSymbolPrice').resolves(price);
    let actual = await api.getSymbolPrice('AUDNZD', true);
    actual.should.match(price);
    _sinon2.default.assert.calledWith(client.getSymbolPrice, 'accountId', 'AUDNZD', true);
  });

  /**
   * @test {MetaApiConnection#getCandle}
   */
  it('should retrieve current candle', async () => {
    let candle = {
      symbol: 'AUDNZD',
      timeframe: '15m',
      time: new Date('2020-04-07T03:45:00.000Z'),
      brokerTime: '2020-04-07 06:45:00.000',
      open: 1.03297,
      high: 1.06309,
      low: 1.02705,
      close: 1.043,
      tickVolume: 1435,
      spread: 17,
      volume: 345
    };
    sandbox.stub(client, 'getCandle').resolves(candle);
    let actual = await api.getCandle('AUDNZD', '15m', true);
    actual.should.match(candle);
    _sinon2.default.assert.calledWith(client.getCandle, 'accountId', 'AUDNZD', '15m', true);
  });

  /**
   * @test {MetaApiConnection#getTick}
   */
  it('should retrieve latest tick', async () => {
    let tick = {
      symbol: 'AUDNZD',
      time: new Date('2020-04-07T03:45:00.000Z'),
      brokerTime: '2020-04-07 06:45:00.000',
      bid: 1.05297,
      ask: 1.05309,
      last: 0.5298,
      volume: 0.13,
      side: 'buy'
    };
    sandbox.stub(client, 'getTick').resolves(tick);
    let actual = await api.getTick('AUDNZD', true);
    actual.should.match(tick);
    _sinon2.default.assert.calledWith(client.getTick, 'accountId', 'AUDNZD', true);
  });

  /**
   * @test {MetaApiConnection#getBook}
   */
  it('should retrieve latest order book', async () => {
    let book = {
      symbol: 'AUDNZD',
      time: new Date('2020-04-07T03:45:00.000Z'),
      brokerTime: '2020-04-07 06:45:00.000',
      book: [{
        type: 'BOOK_TYPE_SELL',
        price: 1.05309,
        volume: 5.67
      }, {
        type: 'BOOK_TYPE_BUY',
        price: 1.05297,
        volume: 3.45
      }]
    };
    sandbox.stub(client, 'getBook').resolves(book);
    let actual = await api.getBook('AUDNZD', true);
    actual.should.match(book);
    _sinon2.default.assert.calledWith(client.getBook, 'accountId', 'AUDNZD', true);
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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