'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _promise = require('babel-runtime/core-js/promise');

var _promise2 = _interopRequireDefault(_promise);

var _values = require('babel-runtime/core-js/object/values');

var _values2 = _interopRequireDefault(_values);

var _entries = require('babel-runtime/core-js/object/entries');

var _entries2 = _interopRequireDefault(_entries);

var _keys = require('babel-runtime/core-js/object/keys');

var _keys2 = _interopRequireDefault(_keys);

var _terminalState = require('./terminalState');

var _terminalState2 = _interopRequireDefault(_terminalState);

var _memoryHistoryStorage = require('./memoryHistoryStorage');

var _memoryHistoryStorage2 = _interopRequireDefault(_memoryHistoryStorage);

var _timeoutError = require('../clients/timeoutError');

var _timeoutError2 = _interopRequireDefault(_timeoutError);

var _randomstring = require('randomstring');

var _randomstring2 = _interopRequireDefault(_randomstring);

var _connectionHealthMonitor = require('./connectionHealthMonitor');

var _connectionHealthMonitor2 = _interopRequireDefault(_connectionHealthMonitor);

var _errorHandler = require('../clients/errorHandler');

var _optionsValidator = require('../clients/optionsValidator');

var _optionsValidator2 = _interopRequireDefault(_optionsValidator);

var _logger = require('../logger');

var _logger2 = _interopRequireDefault(_logger);

var _metaApiConnection = require('./metaApiConnection');

var _metaApiConnection2 = _interopRequireDefault(_metaApiConnection);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Exposes MetaApi MetaTrader streaming API connection to consumers
 */
class StreamingMetaApiConnection extends _metaApiConnection2.default {

  /**
   * Constructs MetaApi MetaTrader streaming Api connection
   * @param {MetaApiWebsocketClient} websocketClient MetaApi websocket client
   * @param {MetatraderAccount} account MetaTrader account id to connect to
   * @param {HistoryStorage} historyStorage terminal history storage. By default an instance of MemoryHistoryStorage
   * will be used.
   * @param {ConnectionRegistry} connectionRegistry metatrader account connection registry
   * @param {Date} [historyStartTime] history start sync time
   * @param {RefreshSubscriptionsOpts} [refreshSubscriptionsOpts] subscriptions refresh options
   */
  constructor(websocketClient, account, historyStorage, connectionRegistry, historyStartTime, refreshSubscriptionsOpts) {
    super(websocketClient, account);
    refreshSubscriptionsOpts = refreshSubscriptionsOpts || {};
    const validator = new _optionsValidator2.default();
    this._minSubscriptionRefreshInterval = validator.validateNonZero(refreshSubscriptionsOpts.minDelayInSeconds, 1, 'refreshSubscriptionsOpts.minDelayInSeconds');
    this._maxSubscriptionRefreshInterval = validator.validateNonZero(refreshSubscriptionsOpts.maxDelayInSeconds, 600, 'refreshSubscriptionsOpts.maxDelayInSeconds');
    this._connectionRegistry = connectionRegistry;
    this._historyStartTime = historyStartTime;
    this._terminalState = new _terminalState2.default();
    this._historyStorage = historyStorage || new _memoryHistoryStorage2.default(account.id, connectionRegistry.application);
    this._healthMonitor = new _connectionHealthMonitor2.default(this);
    this._websocketClient.addSynchronizationListener(account.id, this);
    this._websocketClient.addSynchronizationListener(account.id, this._terminalState);
    this._websocketClient.addSynchronizationListener(account.id, this._historyStorage);
    this._websocketClient.addSynchronizationListener(account.id, this._healthMonitor);
    this._websocketClient.addReconnectListener(this, account.id);
    this._subscriptions = {};
    this._stateByInstanceIndex = {};
    this._refreshMarketDataSubscriptionSessions = {};
    this._refreshMarketDataSubscriptionTimeouts = {};
    this._synchronized = false;
    this._synchronizationListeners = [];
    this._logger = _logger2.default.getLogger('MetaApiConnection');
  }

  /**
   * Opens the connection. Can only be called the first time, next calls will be ignored.
   * @return {Promise} promise resolving when the connection is opened
   */
  async connect() {
    if (!this._opened) {
      this._opened = true;
      try {
        await this.initialize();
        await this.subscribe();
      } catch (err) {
        await this.close();
        throw err;
      }
    }
  }

  /**
   * Clears the order and transaction history of a specified application so that it can be synchronized from scratch
   * (see https://metaapi.cloud/docs/client/websocket/api/removeHistory/).
   * @param {String} [application] application to remove history for
   * @return {Promise} promise resolving when the history is cleared
   */
  removeHistory(application) {
    this._historyStorage.clear();
    return this._websocketClient.removeHistory(this._account.id, application);
  }

  /**
   * Clears the order and transaction history of a specified application and removes application (see
   * https://metaapi.cloud/docs/client/websocket/api/removeApplication/).
   * @return {Promise} promise resolving when the history is cleared and application is removed
   */
  removeApplication() {
    this._historyStorage.clear();
    return this._websocketClient.removeApplication(this._account.id);
  }

  /**
   * Requests the terminal to start synchronization process
   * (see https://metaapi.cloud/docs/client/websocket/synchronizing/synchronize/)
   * @param {String} instanceIndex instance index
   * @returns {Promise} promise which resolves when synchronization started
   */
  async synchronize(instanceIndex) {
    const instance = this.getInstanceNumber(instanceIndex);
    const host = this.getHostName(instanceIndex);
    let startingHistoryOrderTime = new Date(Math.max((this._historyStartTime || new Date(0)).getTime(), (await this._historyStorage.lastHistoryOrderTime(instance)).getTime()));
    let startingDealTime = new Date(Math.max((this._historyStartTime || new Date(0)).getTime(), (await this._historyStorage.lastDealTime(instance)).getTime()));
    let synchronizationId = _randomstring2.default.generate(32);
    this._getState(instanceIndex).lastSynchronizationId = synchronizationId;
    const hashes = this.terminalState.getHashes(this._account.type, instanceIndex);
    return this._websocketClient.synchronize(this._account.id, instance, host, synchronizationId, startingHistoryOrderTime, startingDealTime, hashes.specificationsMd5, hashes.positionsMd5, hashes.ordersMd5);
  }

  /**
   * Initializes meta api connection
   * @return {Promise} promise which resolves when meta api connection is initialized
   */
  async initialize() {
    await this._historyStorage.initialize();
  }

  /**
   * Initiates subscription to MetaTrader terminal
   * @returns {Promise} promise which resolves when subscription is initiated
   */
  async subscribe() {
    if (!this._closed) {
      this._websocketClient.ensureSubscribe(this._account.id);
    }
  }

  /**
   * Subscribes on market data of specified symbol (see
   * https://metaapi.cloud/docs/client/websocket/marketDataStreaming/subscribeToMarketData/).
   * @param {String} symbol symbol (e.g. currency pair or an index)
   * @param {Array<MarketDataSubscription>} subscriptions array of market data subscription to create or update. Please
   * note that this feature is not fully implemented on server-side yet
   * @param {Number} instanceIndex instance index
   * @param {number} [timeoutInSeconds] timeout to wait for prices in seconds, default is 30
   * @returns {Promise} promise which resolves when subscription request was processed
   */
  async subscribeToMarketData(symbol, subscriptions, instanceIndex, timeoutInSeconds) {
    if (!this._terminalState.specification(symbol)) {
      throw new _errorHandler.ValidationError(`Cannot subscribe to market data for symbol ${symbol} because ` + 'symbol does not exist');
    } else {
      subscriptions = subscriptions || [{ type: 'quotes' }];
      if (this._subscriptions[symbol]) {
        const prevSubscriptions = this._subscriptions[symbol].subscriptions;
        subscriptions.forEach(subscription => {
          const index = subscription.type === 'candles' ? prevSubscriptions.findIndex(item => item.type === subscription.type && item.timeframe === subscription.timeframe) : prevSubscriptions.findIndex(item => item.type === subscription.type);
          if (index === -1) {
            prevSubscriptions.push(subscription);
          } else {
            prevSubscriptions[index] = subscription;
          }
        });
      } else {
        this._subscriptions[symbol] = { subscriptions };
      }
      await this._websocketClient.subscribeToMarketData(this._account.id, instanceIndex, symbol, subscriptions);
      return this.terminalState.waitForPrice(symbol, timeoutInSeconds);
    }
  }

  /**
   * Unsubscribes from market data of specified symbol (see
   * https://metaapi.cloud/docs/client/websocket/marketDataStreaming/unsubscribeFromMarketData/).
   * @param {String} symbol symbol (e.g. currency pair or an index)
   * @param {Array<MarketDataUnsubscription>} subscriptions array of subscriptions to cancel
   * @param {Number} instanceIndex instance index
   * @returns {Promise} promise which resolves when unsubscription request was processed
   */
  unsubscribeFromMarketData(symbol, subscriptions, instanceIndex) {
    if (!subscriptions) {
      delete this._subscriptions[symbol];
    } else if (this._subscriptions[symbol]) {
      this._subscriptions[symbol].subscriptions = this._subscriptions[symbol].subscriptions.filter(s => s.type === 'candles' ? !subscriptions.find(s2 => s.type === s2.type && s.timeframe === s2.timeframe) : !subscriptions.find(s2 => s.type === s2.type));
      if (!this._subscriptions[symbol].subscriptions.length) {
        delete this._subscriptions[symbol];
      }
    }
    return this._websocketClient.unsubscribeFromMarketData(this._account.id, instanceIndex, symbol, subscriptions);
  }

  /**
   * Invoked when subscription downgrade has occurred
   * @param {String} instanceIndex index of an account instance connected
   * @param {string} symbol symbol to update subscriptions for
   * @param {Array<MarketDataSubscription>} updates array of market data subscription to update
   * @param {Array<MarketDataUnsubscription>} unsubscriptions array of subscriptions to cancel
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  // eslint-disable-next-line complexity
  async onSubscriptionDowngraded(instanceIndex, symbol, updates, unsubscriptions) {
    let subscriptions = this._subscriptions[symbol];
    if (unsubscriptions && unsubscriptions.length) {
      if (subscriptions) {
        for (let subscription of unsubscriptions) {
          subscriptions = subscriptions.filter(s => s.type === subscription.type);
        }
      }
      this.unsubscribeFromMarketData(symbol, unsubscriptions).catch(err => {
        if (err.name !== _errorHandler.ValidationError) {
          this._logger.error(`${this._account.id}: failed do unsubscribe from market data on subscription downgraded`, err);
        } else {
          this._logger.trace(`${this._account.id}: failed do unsubscribe from market data on subscription downgraded`, err);
        }
      });
    }
    if (updates && updates.length) {
      if (subscriptions) {
        for (let subscription of updates) {
          subscriptions.filter(s => s.type === subscription.type).forEach(s => s.intervalInMilliiseconds = subscription.intervalInMilliseconds);
        }
      }
      this.subscribeToMarketData(symbol, updates).catch(err => {
        this._logger.error(`${this._account.id}: failed do unsubscribe from market data on subscription downgraded`, err);
      });
    }
    if (subscriptions && !subscriptions.length) {
      delete this._subscriptions[symbol];
    }
  }

  /**
   * Returns list of the symbols connection is subscribed to
   * @returns {Array<String>} list of the symbols connection is subscribed to
   */
  get subscribedSymbols() {
    return (0, _keys2.default)(this._subscriptions);
  }

  /**
   * Returns subscriptions for a symbol
   * @param {string} symbol symbol to retrieve subscriptions for
   * @returns {Array<MarketDataSubscription>} list of market data subscriptions for the symbol
   */
  subscriptions(symbol) {
    return (this._subscriptions[symbol] || {}).subscriptions;
  }

  /**
   * Sends client uptime stats to the server.
   * @param {Object} uptime uptime statistics to send to the server
   * @returns {Promise} promise which resolves when uptime statistics is submitted
   */
  saveUptime(uptime) {
    return this._websocketClient.saveUptime(this._account.id, uptime);
  }

  /**
   * Returns local copy of terminal state
   * @returns {TerminalState} local copy of terminal state
   */
  get terminalState() {
    return this._terminalState;
  }

  /**
   * Returns local history storage
   * @returns {HistoryStorage} local history storage
   */
  get historyStorage() {
    return this._historyStorage;
  }

  /**
   * Adds synchronization listener
   * @param {SynchronizationListener} listener synchronization listener to add
   */
  addSynchronizationListener(listener) {
    this._synchronizationListeners.push(listener);
    this._websocketClient.addSynchronizationListener(this._account.id, listener);
  }

  /**
   * Removes synchronization listener for specific account
   * @param {SynchronizationListener} listener synchronization listener to remove
   */
  removeSynchronizationListener(listener) {
    this._synchronizationListeners = this._synchronizationListeners.filter(l => l !== listener);
    this._websocketClient.removeSynchronizationListener(this._account.id, listener);
  }

  /**
   * Invoked when connection to MetaTrader terminal established
   * @param {String} instanceIndex index of an account instance connected
   * @param {Number} replicas number of account replicas launched
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onConnected(instanceIndex, replicas) {
    let key = _randomstring2.default.generate(32);
    let state = this._getState(instanceIndex);
    state.shouldSynchronize = key;
    state.synchronizationRetryIntervalInSeconds = 1;
    state.synchronized = false;
    this._ensureSynchronized(instanceIndex, key);
    let indices = [];
    for (let i = 0; i < replicas; i++) {
      indices.push(i);
    }
    for (let e of (0, _entries2.default)(this._stateByInstanceIndex)) {
      if (!indices.includes(this.getInstanceNumber(e[1].instanceIndex))) {
        delete this._stateByInstanceIndex[e[0]];
      }
    }
  }

  /**
   * Invoked when connection to MetaTrader terminal terminated
   * @param {String} instanceIndex index of an account instance connected
   */
  async onDisconnected(instanceIndex) {
    let state = this._getState(instanceIndex);
    state.lastDisconnectedSynchronizationId = state.lastSynchronizationId;
    state.lastSynchronizationId = undefined;
    state.shouldSynchronize = undefined;
    state.synchronized = false;
    state.disconnected = true;
    const instance = this.getInstanceNumber(instanceIndex);
    delete this._refreshMarketDataSubscriptionSessions[instance];
    clearTimeout(this._refreshMarketDataSubscriptionTimeouts[instance]);
    delete this._refreshMarketDataSubscriptionTimeouts[instance];
  }

  /**
   * Invoked when a synchronization of history deals on a MetaTrader account have finished to indicate progress of an
   * initial terminal state synchronization
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onDealsSynchronized(instanceIndex, synchronizationId) {
    let state = this._getState(instanceIndex);
    state.dealsSynchronized[synchronizationId] = true;
  }

  /**
   * Invoked when a synchronization of history orders on a MetaTrader account have finished to indicate progress of an
   * initial terminal state synchronization
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onHistoryOrdersSynchronized(instanceIndex, synchronizationId) {
    let state = this._getState(instanceIndex);
    state.ordersSynchronized[synchronizationId] = true;
  }

  /**
   * Invoked when connection to MetaApi websocket API restored after a disconnect
   * @return {Promise} promise which resolves when connection to MetaApi websocket API restored after a disconnect
   */
  async onReconnected() {
    this._stateByInstanceIndex = {};
    this._refreshMarketDataSubscriptionSessions = {};
    (0, _values2.default)(this._refreshMarketDataSubscriptionTimeouts).forEach(timeout => clearTimeout(timeout));
    this._refreshMarketDataSubscriptionTimeouts = {};
  }

  /**
   * Invoked when a stream for an instance index is closed
   * @param {String} instanceIndex index of an account instance connected
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onStreamClosed(instanceIndex) {
    delete this._stateByInstanceIndex[instanceIndex];
  }

  /**
   * Invoked when MetaTrader terminal state synchronization is started
   * @param {String} instanceIndex index of an account instance connected
   * @param {Boolean} specificationsUpdated whether specifications are going to be updated during synchronization
   * @param {Boolean} positionsUpdated whether positions are going to be updated during synchronization
   * @param {Boolean} ordersUpdated whether orders are going to be updated during synchronization
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onSynchronizationStarted(instanceIndex, specificationsUpdated, positionsUpdated, ordersUpdated) {
    const instance = this.getInstanceNumber(instanceIndex);
    delete this._refreshMarketDataSubscriptionSessions[instance];
    let sessionId = _randomstring2.default.generate(32);
    this._refreshMarketDataSubscriptionSessions[instance] = sessionId;
    clearTimeout(this._refreshMarketDataSubscriptionTimeouts[instance]);
    delete this._refreshMarketDataSubscriptionTimeouts[instance];
    await this._refreshMarketDataSubscriptions(instance, sessionId);
  }

  /**
   * Returns flag indicating status of state synchronization with MetaTrader terminal
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} synchronizationId optional synchronization request id, last synchronization request id will be used
   * by default
   * @return {Promise<Boolean>} promise resolving with a flag indicating status of state synchronization with MetaTrader
   * terminal
   */
  async isSynchronized(instanceIndex, synchronizationId) {
    return (0, _values2.default)(this._stateByInstanceIndex).reduce((acc, s) => {
      if (instanceIndex !== undefined && s.instanceIndex !== instanceIndex) {
        return acc;
      }
      synchronizationId = synchronizationId || s.lastSynchronizationId;
      let synchronized = !!s.ordersSynchronized[synchronizationId] && !!s.dealsSynchronized[synchronizationId];
      return acc || synchronized;
    }, false);
  }

  /**
   * @typedef {Object} SynchronizationOptions
   * @property {String} [applicationPattern] application regular expression pattern, default is .*
   * @property {String} [synchronizationId] synchronization id, last synchronization request id will be used by
   * default
   * @property {Number} [instanceIndex] index of an account instance to ensure synchronization on, default is to wait
   * for the first instance to synchronize
   * @param {Number} [timeoutInSeconds] wait timeout in seconds, default is 5m
   * @param {Number} [intervalInMilliseconds] interval between account reloads while waiting for a change, default is 1s
   */

  /**
   * Waits until synchronization to MetaTrader terminal is completed
   * @param {SynchronizationOptions} synchronization options
   * @return {Promise} promise which resolves when synchronization to MetaTrader terminal is completed
   * @throws {TimeoutError} if application failed to synchronize with the teminal withing timeout allowed
   */
  // eslint-disable-next-line complexity
  async waitSynchronized(opts) {
    opts = opts || {};
    let instanceIndex = opts.instanceIndex;
    let synchronizationId = opts.synchronizationId;
    let timeoutInSeconds = opts.timeoutInSeconds || 300;
    let intervalInMilliseconds = opts.intervalInMilliseconds || 1000;
    let applicationPattern = opts.applicationPattern || (this._account.application === 'CopyFactory' ? 'CopyFactory.*|RPC' : 'RPC');
    let startTime = Date.now();
    let synchronized;
    while (!(synchronized = await this.isSynchronized(instanceIndex, synchronizationId)) && startTime + timeoutInSeconds * 1000 > Date.now()) {
      await new _promise2.default(res => setTimeout(res, intervalInMilliseconds));
    }
    let state;
    if (instanceIndex === undefined) {
      for (let s of (0, _values2.default)(this._stateByInstanceIndex)) {
        if (await this.isSynchronized(s.instanceIndex, synchronizationId)) {
          state = s;
          instanceIndex = s.instanceIndex;
        }
      }
    } else {
      state = (0, _values2.default)(this._stateByInstanceIndex).find(s => s.instanceIndex === instanceIndex);
    }
    if (!synchronized) {
      throw new _timeoutError2.default('Timed out waiting for MetaApi to synchronize to MetaTrader account ' + this._account.id + ', synchronization id ' + (synchronizationId || state && state.lastSynchronizationId || state && state.lastDisconnectedSynchronizationId));
    }
    let timeLeftInSeconds = Math.max(0, timeoutInSeconds - (Date.now() - startTime) / 1000);
    await this._websocketClient.waitSynchronized(this._account.id, this.getInstanceNumber(instanceIndex), applicationPattern, timeLeftInSeconds);
  }

  /**
   * Closes the connection. The instance of the class should no longer be used after this method is invoked.
   */
  async close() {
    if (!this._closed) {
      this._logger.debug(`${this._account.id}: Closing connection`);
      this._stateByInstanceIndex = {};
      this._connectionRegistry.remove(this._account.id);
      await this._websocketClient.unsubscribe(this._account.id);
      this._websocketClient.removeSynchronizationListener(this._account.id, this);
      this._websocketClient.removeSynchronizationListener(this._account.id, this._terminalState);
      this._websocketClient.removeSynchronizationListener(this._account.id, this._historyStorage);
      this._websocketClient.removeSynchronizationListener(this._account.id, this._healthMonitor);
      for (let listener of this._synchronizationListeners) {
        this._websocketClient.removeSynchronizationListener(this._account.id, listener);
      }
      this._synchronizationListeners = [];
      this._websocketClient.removeReconnectListener(this);
      this._healthMonitor.stop();
      this._refreshMarketDataSubscriptionSessions = {};
      (0, _values2.default)(this._refreshMarketDataSubscriptionTimeouts).forEach(timeout => clearTimeout(timeout));
      this._refreshMarketDataSubscriptionTimeouts = {};
      this._closed = true;
    }
  }

  /**
   * Returns synchronization status
   * @return {boolean} synchronization status
   */
  get synchronized() {
    return (0, _values2.default)(this._stateByInstanceIndex).reduce((acc, s) => acc || s.synchronized, false);
  }

  /**
   * Returns MetaApi account
   * @return {MetatraderAccount} MetaApi account
   */
  get account() {
    return this._account;
  }

  /**
   * Returns connection health monitor instance
   * @return {ConnectionHealthMonitor} connection health monitor instance
   */
  get healthMonitor() {
    return this._healthMonitor;
  }

  async _refreshMarketDataSubscriptions(instanceNumber, session) {
    try {
      if (this._refreshMarketDataSubscriptionSessions[instanceNumber] === session) {
        const subscriptionsList = [];
        (0, _keys2.default)(this._subscriptions).forEach(key => {
          const subscriptions = this.subscriptions(key);
          const subscriptionsItem = { symbol: key };
          if (subscriptions) {
            subscriptionsItem.subscriptions = subscriptions;
          }
          subscriptionsList.push(subscriptionsItem);
        });
        await this._websocketClient.refreshMarketDataSubscriptions(this._account.id, instanceNumber, subscriptionsList);
      }
    } catch (err) {
      this._logger.error(`Error refreshing market data subscriptions job for account ${this._account.id} ` + `${instanceNumber}`, err);
    } finally {
      if (this._refreshMarketDataSubscriptionSessions[instanceNumber] === session) {
        let refreshInterval = (Math.random() * (this._maxSubscriptionRefreshInterval - this._minSubscriptionRefreshInterval) + this._minSubscriptionRefreshInterval) * 1000;
        this._refreshMarketDataSubscriptionTimeouts[instanceNumber] = setTimeout(() => this._refreshMarketDataSubscriptions(instanceNumber, session), refreshInterval);
      }
    }
  }

  _generateStopOptions(stopLoss, takeProfit) {
    let trade = {};
    if (typeof stopLoss === 'number') {
      trade.stopLoss = stopLoss;
    } else if (stopLoss) {
      trade.stopLoss = stopLoss.value;
      trade.stopLossUnits = stopLoss.units;
    }
    if (typeof takeProfit === 'number') {
      trade.takeProfit = takeProfit;
    } else if (takeProfit) {
      trade.takeProfit = takeProfit.value;
      trade.takeProfitUnits = takeProfit.units;
    }
    return trade;
  }

  async _ensureSynchronized(instanceIndex, key) {
    let state = this._getState(instanceIndex);
    if (state && !this._closed) {
      try {
        const synchronizationResult = await this.synchronize(instanceIndex);
        if (synchronizationResult) {
          state.synchronized = true;
          state.synchronizationRetryIntervalInSeconds = 1;
        }
      } catch (err) {
        this._logger.error('MetaApi websocket client for account ' + this._account.id + ':' + instanceIndex + ' failed to synchronize', err);
        if (state.shouldSynchronize === key) {
          setTimeout(this._ensureSynchronized.bind(this, instanceIndex, key), state.synchronizationRetryIntervalInSeconds * 1000);
          state.synchronizationRetryIntervalInSeconds = Math.min(state.synchronizationRetryIntervalInSeconds * 2, 300);
        }
      }
    }
  }

  _getState(instanceIndex) {
    if (!this._stateByInstanceIndex['' + instanceIndex]) {
      this._stateByInstanceIndex['' + instanceIndex] = {
        instanceIndex,
        ordersSynchronized: {},
        dealsSynchronized: {},
        shouldSynchronize: undefined,
        synchronizationRetryIntervalInSeconds: 1,
        synchronized: false,
        lastDisconnectedSynchronizationId: undefined,
        lastSynchronizationId: undefined,
        disconnected: false
      };
    }
    return this._stateByInstanceIndex['' + instanceIndex];
  }

}
exports.default = StreamingMetaApiConnection;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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