'use strict';

var _should = require('should');

var _should2 = _interopRequireDefault(_should);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _terminalState = require('./terminalState');

var _terminalState2 = _interopRequireDefault(_terminalState);

var _cryptoJs = require('crypto-js');

var _cryptoJs2 = _interopRequireDefault(_cryptoJs);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * @test {TerminalState}
 */
describe('TerminalState', () => {

  let state;
  const md5 = arg => _cryptoJs2.default.MD5(arg).toString();

  beforeEach(() => {
    state = new _terminalState2.default();
  });

  /**
   * @test {TerminalState#onConnected}
   * @test {TerminalState#onDisconnected}
   * @test {TerminalState#connected}
   */
  it('should return connection state', () => {
    state.connected.should.be.false();
    state.onConnected('1:ps-mpa-1');
    state.connected.should.be.true();
    state.onDisconnected('1:ps-mpa-1');
    state.connected.should.be.false();
  });

  /**
   * @test {TerminalState#onBrokerConnectionStatus}
   * @test {TerminalState#connectedToBroker}
   */
  it('should return broker connection state', async () => {
    const clock = _sinon2.default.useFakeTimers();
    state.connectedToBroker.should.be.false();
    state.onBrokerConnectionStatusChanged('1:ps-mpa-1', true);
    state.connectedToBroker.should.be.true();
    state.onBrokerConnectionStatusChanged('1:ps-mpa-1', false);
    state.connectedToBroker.should.be.false();
    state.onBrokerConnectionStatusChanged('1:ps-mpa-1', true);
    state.onDisconnected('1:ps-mpa-1');
    state.connectedToBroker.should.be.false();
    await clock.tickAsync(65000);
    clock.restore();
  });

  /**
   * @test {TerminalState#onAccountInformationUpdated}
   * @test {TerminalState#accountInformation}
   */
  it('should return account information', () => {
    _should2.default.not.exist(state.accountInformation);
    state.onAccountInformationUpdated('1:ps-mpa-1', { balance: 1000 });
    state.accountInformation.should.match({ balance: 1000 });
  });

  /**
   * @test {TerminalState#onPositionUpdated}
   * @test {TerminalState#onPositionRemoved}
   * @test {TerminalState#positions}
   */
  it('should return positions', () => {
    state.positions.length.should.equal(0);
    state.onPositionUpdated('1:ps-mpa-1', { id: '1', profit: 10 });
    state.onPositionUpdated('1:ps-mpa-1', { id: '2' });
    state.onPositionUpdated('1:ps-mpa-1', { id: '1', profit: 11 });
    state.onPositionRemoved('1:ps-mpa-1', '2');
    state.positions.length.should.equal(1);
    state.positions.should.match([{ id: '1', profit: 11 }]);
  });

  /**
   * @test {TerminalState#onPendingOrderUpdated}
   * @test {TerminalState#onPendingOrderCompleted}
   * @test {TerminalState#orders}
   */
  it('should return orders', async () => {
    state.orders.length.should.equal(0);
    await state.onPendingOrderUpdated('1:ps-mpa-1', { id: '1', openPrice: 10 });
    await state.onPendingOrderUpdated('1:ps-mpa-1', { id: '2' });
    await state.onPendingOrderUpdated('1:ps-mpa-1', { id: '1', openPrice: 11 });
    state.orders.length.should.equal(2);
    await state.onPendingOrderCompleted('1:ps-mpa-1', '2');
    state.orders.length.should.equal(1);
    state.orders.should.match([{ id: '1', openPrice: 11 }]);
  });

  /**
   * @test {TerminalState#onSymbolSpecificationsUpdated}
   * @test {TerminalState#specifications}
   * @test {TerminalState#specification}
   */
  it('should return specifications', () => {
    state.specifications.length.should.equal(0);
    state.onSymbolSpecificationsUpdated('1:ps-mpa-1', [{ symbol: 'EURUSD', tickSize: 0.00001 }, { symbol: 'GBPUSD' }], []);
    state.onSymbolSpecificationsUpdated('1:ps-mpa-1', [{ symbol: 'AUDNZD' }, { symbol: 'EURUSD',
      tickSize: 0.0001 }], ['AUDNZD']);
    state.specifications.length.should.equal(2);
    state.specifications.should.match([{ symbol: 'EURUSD', tickSize: 0.0001 }, { symbol: 'GBPUSD' }]);
    state.specification('EURUSD').should.match({ symbol: 'EURUSD', tickSize: 0.0001 });
  });

  /**
   * @test {TerminalState#onSymbolPricesUpdated}
   * @test {TerminalState#price}
   */
  it('should return price', () => {
    _should2.default.not.exist(state.price('EURUSD'));
    state.onSymbolPricesUpdated('1:ps-mpa-1', [{ time: new Date(), symbol: 'EURUSD', bid: 1, ask: 1.1 }]);
    state.onSymbolPricesUpdated('1:ps-mpa-1', [{ time: new Date(), symbol: 'GBPUSD' }]);
    state.onSymbolPricesUpdated('1:ps-mpa-1', [{ time: new Date(), symbol: 'EURUSD', bid: 1, ask: 1.2 }]);
    state.price('EURUSD').should.match({ symbol: 'EURUSD', bid: 1, ask: 1.2 });
  });

  /**
   * @test {TerminalState#onSymbolPricesUpdated}
   * @test {TerminalState#price}
   */
  it('should wait for price', async () => {
    _should2.default.not.exist(state.price('EURUSD'));
    let promise = state.waitForPrice('EURUSD');
    state.onSymbolPricesUpdated('1:ps-mpa-1', [{ time: new Date(), symbol: 'EURUSD', bid: 1, ask: 1.1 }]);
    (await promise).should.match({ symbol: 'EURUSD', bid: 1, ask: 1.1 });
  });

  /**
   * @test {TerminalState#onSymbolPricesUpdated}
   * @test {TerminalState#accountInformation}
   * @test {TerminalState#positions}
   */
  it('should update account equity and position profit on price update', () => {
    state.onAccountInformationUpdated('1:ps-mpa-1', { equity: 1000, balance: 800 });
    state.onPositionsReplaced('1:ps-mpa-1', [{
      id: '1',
      symbol: 'EURUSD',
      type: 'POSITION_TYPE_BUY',
      currentPrice: 9,
      currentTickValue: 0.5,
      openPrice: 8,
      profit: 100,
      volume: 2
    }]);
    state.onPendingOrdersSynchronized('1:ps-mpa-1', 'synchronizationId');
    state.onPositionUpdated('1:ps-mpa-1', {
      id: '2',
      symbol: 'AUDUSD',
      type: 'POSITION_TYPE_BUY',
      currentPrice: 9,
      currentTickValue: 0.5,
      openPrice: 8,
      profit: 100,
      volume: 2
    });
    state.onPositionsSynchronized('1:ps-mpa-1', 'synchronizationId');
    state.onSymbolSpecificationsUpdated('1:ps-mpa-1', [{ symbol: 'EURUSD', tickSize: 0.01, digits: 5 }, { symbol: 'AUDUSD', tickSize: 0.01, digits: 5 }], []);
    state.onSymbolPricesUpdated('1:ps-mpa-1', [{
      time: new Date(),
      symbol: 'EURUSD',
      profitTickValue: 0.5,
      lossTickValue: 0.5,
      bid: 10,
      ask: 11
    }, {
      time: new Date(),
      symbol: 'AUDUSD',
      profitTickValue: 0.5,
      lossTickValue: 0.5,
      bid: 10,
      ask: 11
    }]);
    state.positions.map(p => p.profit).should.match([200, 200]);
    state.positions.map(p => p.unrealizedProfit).should.match([200, 200]);
    state.positions.map(p => p.currentPrice).should.match([10, 10]);
    state.accountInformation.equity.should.equal(1200);
  });

  /**
   * @test {TerminalState#onSymbolPricesUpdated}
   * @test {TerminalState#accountInformation}
   * @test {TerminalState#positions}
   */
  it('should update margin fields on price update', () => {
    state.onAccountInformationUpdated('1:ps-mpa-1', { equity: 1000, balance: 800 });
    state.onSymbolPricesUpdated('1:ps-mpa-1', [], 100, 200, 400, 40000);
    state.accountInformation.equity.should.equal(100);
    state.accountInformation.margin.should.equal(200);
    state.accountInformation.freeMargin.should.equal(400);
    state.accountInformation.marginLevel.should.equal(40000);
  });

  /**
   * @test {TerminalState#onSymbolPriceUpdated}
   * @test {TerminalState#orders}
   */
  it('should update order currentPrice on price update', () => {
    state.onPendingOrderUpdated('1:ps-mpa-1', {
      id: '1',
      symbol: 'EURUSD',
      type: 'ORDER_TYPE_BUY_LIMIT',
      currentPrice: 9
    });
    state.onPendingOrderUpdated('1:ps-mpa-1', {
      id: '2',
      symbol: 'AUDUSD',
      type: 'ORDER_TYPE_SELL_LIMIT',
      currentPrice: 9
    });
    state.onSymbolSpecificationsUpdated('1:ps-mpa-1', [{ symbol: 'EURUSD', tickSize: 0.01 }], []);
    state.onSymbolPricesUpdated('1:ps-mpa-1', [{
      time: new Date(),
      symbol: 'EURUSD',
      profitTickValue: 0.5,
      lossTickValue: 0.5,
      bid: 10,
      ask: 11
    }]);
    state.orders.map(o => o.currentPrice).should.match([11, 9]);
  });

  /**
   * @test {TerminalState#onStreamClosed}
   */
  it('should remove state on closed stream', async () => {
    const date = new Date();
    _sinon2.default.assert.match(state.price('EURUSD'), undefined);
    await state.onSymbolPricesUpdated('1:ps-mpa-1', [{ time: date, symbol: 'EURUSD', bid: 1, ask: 1.1 }]);
    state.onPendingOrdersSynchronized('1:ps-mpa-1', 'synchronizationId');
    _sinon2.default.assert.match(state.price('EURUSD'), { time: date, symbol: 'EURUSD', bid: 1, ask: 1.1 });
    await state.onStreamClosed('1:ps-mpa-1');
  });

  /**
   * @test {TerminalState#onSynchronizationStarted}
   */
  it('should reset state on synchronization started event', async () => {
    const specification = { symbol: 'EURUSD', tickSize: 0.01 };
    const positions = [{
      id: '1',
      symbol: 'EURUSD',
      type: 'POSITION_TYPE_BUY',
      currentPrice: 9,
      currentTickValue: 0.5,
      openPrice: 8,
      profit: 100,
      volume: 2
    }];
    const orders = [{
      id: '1',
      symbol: 'EURUSD',
      type: 'ORDER_TYPE_BUY_LIMIT',
      currentPrice: 9
    }];
    await state.onAccountInformationUpdated('1:ps-mpa-1', { 'balance': 1000 });
    await state.onSymbolSpecificationsUpdated('1:ps-mpa-1', [specification], []);
    await state.onPositionsReplaced('1:ps-mpa-1', positions);
    await state.onPendingOrdersReplaced('1:ps-mpa-1', orders);
    _sinon2.default.assert.match(state.accountInformation, { balance: 1000 });
    _sinon2.default.assert.match(state.specification('EURUSD'), specification);
    await state.onSynchronizationStarted('1:ps-mpa-1', false, false, false);
    state.onPendingOrdersSynchronized('1:ps-mpa-1', 'synchronizationId');
    _sinon2.default.assert.match(state.accountInformation, undefined);
    _sinon2.default.assert.match(state.specification('EURUSD'), specification);
    _sinon2.default.assert.match(state.orders, orders);
    _sinon2.default.assert.match(state.positions, positions);
    await state.onSynchronizationStarted('1:ps-mpa-1', true, false, false);
    state.onPendingOrdersSynchronized('1:ps-mpa-1', 'synchronizationId');
    _sinon2.default.assert.match(state.specification('EURUSD'), undefined);
    _sinon2.default.assert.match(state.orders, orders);
    _sinon2.default.assert.match(state.positions, positions);
    await state.onSynchronizationStarted('1:ps-mpa-1', true, false, true);
    state.onPendingOrdersSynchronized('1:ps-mpa-1', 'synchronizationId');
    _sinon2.default.assert.match(state.orders, []);
    _sinon2.default.assert.match(state.positions, positions);
    await state.onSynchronizationStarted('1:ps-mpa-1', true, true, true);
    state.onPendingOrdersSynchronized('1:ps-mpa-1', 'synchronizationId');
    _sinon2.default.assert.match(state.positions, []);
  });

  /**
   * @test {TerminalState#getHashes}
   */
  it('should return hashes for terminal state data for cloud-g1 accounts', async () => {
    const specificationsHash = md5('[{"symbol":"AUDNZD","tickSize":0.01000000},{"symbol":"EURUSD",' + '"tickSize":0.00000100,"contractSize":1.00000000,"maxVolume":30000.00000000,' + '"hedgedMarginUsesLargerLeg":false,"digits":3}]');
    const positionsHash = md5('[{"id":"46214692","type":"POSITION_TYPE_BUY","symbol":"GBPUSD","magic":1000,' + '"openPrice":1.26101000,"volume":0.07000000,"swap":0.00000000,"commission":-0.25000000,' + '"stopLoss":1.17721000}]');
    const ordersHash = md5('[{"id":"46871284","type":"ORDER_TYPE_BUY_LIMIT","state":"ORDER_STATE_PLACED",' + '"symbol":"AUDNZD","magic":123456,"platform":"mt5","openPrice":1.03000000,' + '"volume":0.01000000,"currentVolume":0.01000000}]');
    let hashes = state.getHashes('cloud-g1', '1:ps-mpa-1');
    _sinon2.default.assert.match(hashes.specificationsMd5, null);
    _sinon2.default.assert.match(hashes.positionsMd5, null);
    _sinon2.default.assert.match(hashes.ordersMd5, null);
    await state.onSymbolSpecificationsUpdated('1:ps-mpa-1', [{ symbol: 'AUDNZD', tickSize: 0.01, description: 'Test1' }, { symbol: 'EURUSD', tickSize: 0.000001, contractSize: 1, maxVolume: 30000,
      hedgedMarginUsesLargerLeg: false, digits: 3, description: 'Test2' }], []);
    await state.onPositionsReplaced('1:ps-mpa-1', [{
      id: '46214692',
      type: 'POSITION_TYPE_BUY',
      symbol: 'GBPUSD',
      magic: 1000,
      time: new Date('2020-04-15T02:45:06.521Z'),
      updateTime: new Date('2020-04-15T02:45:06.521Z'),
      openPrice: 1.26101,
      currentPrice: 1.24883,
      currentTickValue: 1,
      volume: 0.07,
      swap: 0,
      profit: -85.25999999999966,
      commission: -0.25,
      clientId: 'TE_GBPUSD_7hyINWqAlE',
      stopLoss: 1.17721,
      unrealizedProfit: -85.25999999999901,
      realizedProfit: -6.536993168992922e-13,
      updateSequenceNumber: 13246,
      accountCurrencyExchangeRate: 1,
      comment: 'test',
      brokerComment: 'test2'
    }]);
    await state.onPendingOrdersReplaced('1:ps-mpa-1', [{
      id: '46871284',
      type: 'ORDER_TYPE_BUY_LIMIT',
      state: 'ORDER_STATE_PLACED',
      symbol: 'AUDNZD',
      magic: 123456,
      platform: 'mt5',
      time: '2020-04-20T08:38:58.270Z',
      openPrice: 1.03,
      currentPrice: 1.05206,
      volume: 0.01,
      currentVolume: 0.01,
      comment: 'COMMENT2',
      updateSequenceNumber: 13246,
      accountCurrencyExchangeRate: 1,
      brokerComment: 'test2',
      clientId: 'TE_GBPUSD_7hyINWqAlE'
    }]);
    state.onPendingOrdersSynchronized('1:ps-mpa-1', 'synchronizationId');
    hashes = state.getHashes('cloud-g1', '1:ps-mpa-1');
    _sinon2.default.assert.match(hashes.specificationsMd5, specificationsHash);
    _sinon2.default.assert.match(hashes.positionsMd5, positionsHash);
    _sinon2.default.assert.match(hashes.ordersMd5, ordersHash);
  });

  /**
   * @test {TerminalState#getHashes}
   */
  it('should return hashes for terminal state data for cloud-g2 accounts', async () => {
    const specificationsHash = md5('[{"symbol":"AUDNZD","tickSize":0.01,"description":"Test1"},' + '{"symbol":"EURUSD","tickSize":0.000001,"contractSize":1,"maxVolume":30000,' + '"hedgedMarginUsesLargerLeg":false,"digits":3,"description":"Test2"}]');
    const positionsHash = md5('[{"id":"46214692","type":"POSITION_TYPE_BUY","symbol":"GBPUSD","magic":1000,' + '"time":"2020-04-15T02:45:06.521Z","updateTime":"2020-04-15T02:45:06.521Z","openPrice":1.26101,' + '"volume":0.07,"swap":0,"commission":-0.25,"stopLoss":1.17721}]');
    const ordersHash = md5('[{"id":"46871284","type":"ORDER_TYPE_BUY_LIMIT","state":"ORDER_STATE_PLACED",' + '"symbol":"AUDNZD","magic":123456,"platform":"mt5","time":"2020-04-20T08:38:58.270Z","openPrice":1.03,' + '"volume":0.01,"currentVolume":0.01}]');
    let hashes = state.getHashes('cloud-g2', '1:ps-mpa-1');
    _sinon2.default.assert.match(hashes.specificationsMd5, null);
    _sinon2.default.assert.match(hashes.positionsMd5, null);
    _sinon2.default.assert.match(hashes.ordersMd5, null);
    await state.onSymbolSpecificationsUpdated('1:ps-mpa-1', [{ symbol: 'AUDNZD', tickSize: 0.01, description: 'Test1' }, { symbol: 'EURUSD', tickSize: 0.000001, contractSize: 1, maxVolume: 30000,
      hedgedMarginUsesLargerLeg: false, digits: 3, description: 'Test2' }], []);
    await state.onPositionsReplaced('1:ps-mpa-1', [{
      id: '46214692',
      type: 'POSITION_TYPE_BUY',
      symbol: 'GBPUSD',
      magic: 1000,
      time: new Date('2020-04-15T02:45:06.521Z'),
      updateTime: new Date('2020-04-15T02:45:06.521Z'),
      openPrice: 1.26101,
      currentPrice: 1.24883,
      currentTickValue: 1,
      volume: 0.07,
      swap: 0,
      profit: -85.25999999999966,
      commission: -0.25,
      clientId: 'TE_GBPUSD_7hyINWqAlE',
      stopLoss: 1.17721,
      unrealizedProfit: -85.25999999999901,
      realizedProfit: -6.536993168992922e-13,
      updateSequenceNumber: 13246,
      accountCurrencyExchangeRate: 1,
      comment: 'test',
      brokerComment: 'test2'
    }]);
    await state.onPendingOrdersReplaced('1:ps-mpa-1', [{
      id: '46871284',
      type: 'ORDER_TYPE_BUY_LIMIT',
      state: 'ORDER_STATE_PLACED',
      symbol: 'AUDNZD',
      magic: 123456,
      platform: 'mt5',
      time: '2020-04-20T08:38:58.270Z',
      openPrice: 1.03,
      currentPrice: 1.05206,
      volume: 0.01,
      currentVolume: 0.01,
      comment: 'COMMENT2',
      updateSequenceNumber: 13246,
      accountCurrencyExchangeRate: 1,
      brokerComment: 'test2',
      clientId: 'TE_GBPUSD_7hyINWqAlE'
    }]);
    state.onPendingOrdersSynchronized('1:ps-mpa-1', 'synchronizationId');
    hashes = state.getHashes('cloud-g2', '1:ps-mpa-1');
    _sinon2.default.assert.match(hashes.specificationsMd5, specificationsHash);
    _sinon2.default.assert.match(hashes.positionsMd5, positionsHash);
    _sinon2.default.assert.match(hashes.ordersMd5, ordersHash);
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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