'use strict';

import SynchronizationListener from '../clients/metaApi/synchronizationListener';

/**
 * Abstract class which defines MetaTrader history storage interface.
 */
export default class HistoryStorage extends SynchronizationListener {

  /**
   * Constructs the history storage
   */
  constructor() {
    super();
    this._orderSynchronizationFinished = {};
    this._dealSynchronizationFinished = {};
  }

  /**
   * Initializes the storage and loads required data from a persistent storage
   */
  async initialize() {}

  /**
   * Returns flag indicating whether order history synchronization have finished
   * @return {Boolean} flag indicating whether order history synchronization have finished
   */
  get orderSynchronizationFinished() {
    return Object.values(this._orderSynchronizationFinished).reduce((acc, r) => acc || r, false);
  }

  /**
   * Returns flag indicating whether deal history synchronization have finished
   * @return {Boolean} flag indicating whether deal history synchronization have finished
   */
  get dealSynchronizationFinished() {
    return Object.values(this._dealSynchronizationFinished).reduce((acc, r) => acc || r, false);
  }

  /**
   * Clears the storage and deletes persistent data
   */
  async clear(){
    throw Error('Abstract method clear has no implementation');
  }

  /**
   * Returns the time of the last history order record stored in the history storage
   * @param {String} [instanceIndex] index of an account instance connected
   * @returns {Date} the time of the last history order record stored in the history storage
   */
  async lastHistoryOrderTime(instanceIndex) {
    throw Error('Abstract method lastHistoryOrderTime has no implementation');
  }

  /**
   * Returns the time of the last history deal record stored in the history storage
   * @param {String} [instanceIndex] index of an account instance connected
   * @returns {Date} the time of the last history deal record stored in the history storage
   */
  async lastDealTime(instanceIndex) {
    throw Error('Abstract method lastDealTime has no implementation');
  }

  /**
   * Invoked when a new MetaTrader history order is added
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderOrder} historyOrder new MetaTrader history order
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onHistoryOrderAdded(instanceIndex, historyOrder) {
    throw Error('Abstract method onHistoryOrderAdded has no implementation');
  }

  /**
   * Invoked when a new MetaTrader history deal is added
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderDeal} deal new MetaTrader history deal
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onDealAdded(instanceIndex, deal) {
    throw Error('Abstract method onDealAdded has no implementation');
  }

  /**
   * Invoked when a synchronization of history deals on a MetaTrader account have finished to indicate progress of an
   * initial terminal state synchronization
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onDealsSynchronized(instanceIndex, synchronizationId) {
    const instance = this.getInstanceNumber(instanceIndex);
    this._dealSynchronizationFinished['' + instance] = true;
  }

  /**
   * Invoked when a synchronization of history orders on a MetaTrader account have finished to indicate progress of an
   * initial terminal state synchronization
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onHistoryOrdersSynchronized(instanceIndex, synchronizationId) {
    const instance = this.getInstanceNumber(instanceIndex);
    this._orderSynchronizationFinished['' + instance] = true;
  }

  /**
   * Invoked when connection to MetaTrader terminal established
   * @param {String} instanceIndex index of an account instance connected
   */
  onConnected(instanceIndex) {
    const instance = this.getInstanceNumber(instanceIndex);
    this._orderSynchronizationFinished['' + instance] = false;
    this._dealSynchronizationFinished['' + instance] = false;
  }

}
